package com.priusis.controller;

import cn.hutool.core.io.unit.DataSizeUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.system.oshi.OshiUtil;
import com.priusis.lib.R;
import com.priusis.util.MacAddrUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.PostConstruct;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Map;

/**
 * @author yangli
 * @since 2021/09/06
 */
@Slf4j
@RestController
@RequestMapping(value = "device_info")
public class DeviceInfoController {

    private Map<String, Object> info;

    @GetMapping(value = "get")
    public R<Map<String, Object>> get() throws UnknownHostException, SocketException {
        if(info == null) refresh();
        return R.success(info);
    }

    @Scheduled(fixedDelay = 5000)
    protected void refresh() throws UnknownHostException {
        info = MapUtil.<String, Object>builder()
                .put("operatingSystem", OshiUtil.getOs().toString())
                .put("mac", MacAddrUtil.get())
                .put("baseboard", OshiUtil.getSystem().getBaseboard().getManufacturer() + " " + OshiUtil.getSystem().getBaseboard().getVersion())
                .put("ip", InetAddress.getLocalHost().getHostAddress())
                .put("cpuModel", OshiUtil.getCpuInfo(0).getCpuModel())
                .put("cpuTemp", OshiUtil.getSensors().getCpuTemperature())
                .put("memoryCap", DataSizeUtil.format(OshiUtil.getMemory().getTotal()))
                .put("diskModel", OshiUtil.getHardware().getDiskStores()[0].getModel())
                .put("diskCap", DataSizeUtil.format(OshiUtil.getHardware().getDiskStores()[0].getSize()))
                .build();
    }

    @PostConstruct
    public void init() {
        try {
            refresh();
        } catch (UnknownHostException e) {
            log.error("初始化失败", e);
        }
    }

    public static void main(String[] args) throws UnknownHostException {
        System.out.println("操作系统");
        System.out.println(OshiUtil.getOs().toString());
        System.out.println("MAC地址");
        System.out.println(MacAddrUtil.get());
        System.out.println("主板");
        System.out.println(OshiUtil.getSystem().getBaseboard().getManufacturer() + " " + OshiUtil.getSystem().getBaseboard().getVersion());
        System.out.println("IP");
        System.out.println(InetAddress.getLocalHost().getHostAddress());
        System.out.println("CPU型号");
        System.out.println(OshiUtil.getCpuInfo(0).getCpuModel());
        System.out.println("CPU温度");
        System.out.println(OshiUtil.getSensors().getCpuTemperature());
        System.out.println("内存容量");
        System.out.println(DataSizeUtil.format(OshiUtil.getMemory().getTotal()));
        System.out.println("硬盘型号");
        System.out.println(OshiUtil.getHardware().getDiskStores()[0].getModel());
        System.out.println("硬盘容量");
        System.out.println(DataSizeUtil.format(OshiUtil.getHardware().getDiskStores()[0].getSize()));
    }
}
