package com.priusis.utils;

import com.sun.jna.Library;
import com.sun.jna.Native;
import com.sun.jna.Platform;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.system.ApplicationHome;

import java.io.File;
import java.io.UnsupportedEncodingException;


/******************************************************************************************************************************
 * 调用流程举例:
 *
 *		// 定义回调函数
 *		void device_change_cb_fn(int changeType, char *devInfoBuf)
 *        {
 *			printf("=== get cb type=%d, data=[%s]\r\n", changeType, devInfoBuf);
 *        }
 *
 *
 *		// --- 1. 初始化 ---
 *		// 初始化接口
 *		APQ_HWPORT_Initialize()
 *
 *		// 注册设备热插拔通知后，接收热插拔事件
 *		APQ_HWPORT_Register_Device_Notification(device_change_cb_fn)
 *
 *		// --- 2. 权限操作 ---
 *		// 可选：获取某个硬件接口权限
 *		APQ_HWPORT_get_normal_device_permit()
 *
 *		// 设置某硬件接口权限
 *		APQ_HWPORT_set_normal_device_permit()
 *		APQ_HWPORT_set_storage_device_permit()
 *		// 设置网络限速，将启动限速线程
 *       APQ_HWPORT_set_net_device_permit(APQ_HWPORT_NET_DEVID_ETH_CARD, APQ_HWPORT_NET_PERM_LIMIT, 0, NULL);
 *		// 应用配置
 *		APQ_HWPORT_Apply()
 *
 *		// 解除网络限速，清理限速线程
 * 		APQ_HWPORT_set_net_device_permit(APQ_HWPORT_NET_DEVID_ETH_CARD, APQ_HWPORT_NET_PERM_NO_LIMIT, 0, NULL);
 *		// 应用配置
 *		APQ_HWPORT_Apply()
 *
 *		// --- 3. 清理资源 ---
 *		// 停止监听并释放资源
 *		APQ_HWPORT_Unregister_Device_Notification()
 *
 *		// 清理资源
 *		APQ_HWPORT_Uninitialize()
 *
 ******************************************************************************************************************************/
public interface IHwPortController extends Library {

    //IHwPortController instanceDll = (IHwPortController) Native.loadLibrary(ParseUrlUtil.parseAbsolutePath("/dact/x64/Release/HwPortController.dll", System.getProperty("user.dir") + "\\HwPortController.dll"), IHwPortController.class);
    IHwPortController instanceDll = (IHwPortController) Native.loadLibrary("D:\\work\\priusis\\priusis-iot\\apq-iot\\apq-client\\apq-pc-control\\src\\main\\resources\\dact\\x64\\Release\\HwPortController.dll", IHwPortController.class);

    /**
     * 初始化硬件接口管理模块
     * 备  注：调用其他接口前必须调用一次此接口初始化资源
     *
     * @return 成功返回 APQ_HWPORT_RESULT_OK，失败返回 APQ_HWPORT_RESULT_ERROR_开头的值  {@link  HwportResultEnum}
     */
    int APQ_HWPORT_Initialize();

    /**
     * 应用硬件接口管理模块的配置内容
     * 备  注：调用完成此接口后，设置才会生效
     *
     * @return 成功返回 APQ_HWPORT_RESULT_OK，失败返回 APQ_HWPORT_RESULT_ERROR_开头的值  {@link  HwportResultEnum}
     */
    int APQ_HWPORT_Apply();

    /**
     * 清理硬件接口管理模块
     * 备  注：最后必须调用此函数清理资源
     *
     * @return 成功返回 APQ_HWPORT_RESULT_OK，失败返回 APQ_HWPORT_RESULT_ERROR_开头的值  {@link  HwportResultEnum}
     */
    int APQ_HWPORT_Uninitialize();

    /**
     * 获取普通设备权限
     *
     * @param device_id 指定普通设备硬件接口类型 {@link  HwportNormalDevidEnum}
     * @return 返回 {@link HwportNormalPermEnum} 权限类型
     */
    int APQ_HWPORT_get_normal_device_permit(int device_id);

    /**
     * 设置普通设备权限
     *
     * @param device_id 枚举类型设备ID号 {@link HwportNormalDevidEnum}
     * @param perm      普通类权限 {@link HwportNormalPermEnum}
     * @param data      预留参数字串，未明确的传NULL
     * @return 成功返回 APQ_HWPORT_RESULT_OK，失败返回 APQ_HWPORT_RESULT_ERROR_开头的值 {@link  HwportResultEnum}
     */
    int APQ_HWPORT_set_normal_device_permit(int device_id, int perm, char[] data);

    /**
     * 获取可移动存储设备权限
     *
     * @param device_id 指定可移动存储设备硬件接口类型 {@link  HwportStorageDevidEnum}
     * @return 返回 {@link HwportStoragePermEnum} 权限类型
     */
    int APQ_HWPORT_get_storage_device_permit(int device_id);

    /**
     * 设置可移动存储设备权限
     *
     * @param device_id 枚举类型设备ID号 {@link HwportStorageDevidEnum}
     * @param perm      存储类权限 {@link HwportStoragePermEnum}
     * @param data      预留参数字串，未明确的传NULL
     * @return 成功返回 APQ_HWPORT_RESULT_OK，失败返回 APQ_HWPORT_RESULT_ERROR_开头的值 {@link  HwportResultEnum}
     */
    int APQ_HWPORT_set_storage_device_permit(int device_id, int perm, char[] data);

    /**
     * 获取网络设备权限
     *
     * @param device_id 指定网络设备硬件接口类型 {@link  HwportNetDevidEnum}
     * @return 返回存放配置内的带宽限速值 {@link HwportNetPermEnum} 权限类型
     */
    int APQ_HWPORT_get_net_device_permit(int device_id, int speed);

    /**
     * 获取网络设备权限
     *
     * @param device_id 指定网络设备硬件接口类型 {@link  HwportNetDevidEnum}
     * @return 返回存放配置内的带宽限速值 {@link HwportNetPermEnum} 权限类型
     */
    int APQ_HWPORT_get_net_device_permit(int device_id);

    /**
     * 设置网络设备权限
     * 设置限速选项会启动限速线程，退出程序时需要调用APQ_HWPORT_NET_PERM_NO_LIMIT或者其他不限速的权限清理限速线程
     *
     * @param device_id 枚举类型设备ID号 {@link HwportNetDevidEnum}
     * @param perm      存储类权限 {@link HwportNetPermEnum}
     * @param speed     带宽限制值，单位KB/s
     * @param data      预留参数字串，未明确的传NULL
     * @return 成功返回 APQ_HWPORT_RESULT_OK，失败返回 APQ_HWPORT_RESULT_ERROR_开头的值 {@link  HwportResultEnum}
     */
    int APQ_HWPORT_set_net_device_permit(int device_id, int perm, int speed, char[] data);

    /**
     * 注册设备热插拔通知（如USB设备）
     * 备  注：不使用时，需要调用 APQ_HWPORT_Unregister_Device_Notification 停止监听并释放资源
     *
     * @param P_DEVICE_CHANGE_CB callback_fn: 		指定事件发生时的回调函数
     * @return 成功返回 APQ_HWPORT_RESULT_OK，失败返回 APQ_HWPORT_RESULT_ERROR_开头的值 {@link  HwportResultEnum}
     */
    //int APQ_HWPORT_Register_Device_Notification(IN P_DEVICE_CHANGE_CB callback_fn)

    /**
     * 停止监听并释放设备热插拔通知相关资源
     *
     * @return 成功返回 APQ_HWPORT_RESULT_OK，失败返回 APQ_HWPORT_RESULT_ERROR_开头的值 {@link  HwportResultEnum}
     */
    int APQ_HWPORT_Unregister_Device_Notification();

    /**
     * 根据设备类型和所属设备的权限类型获取权限字符名，用于打印结果
     *
     * @param device_type {@link  HwportDeviceTypeEnum}
     * @param devid_type  {@link  HwportStorageDevidEnum} {@link  HwportNetDevidEnum} {@link  HwportNormalDevidEnum}
     * @return 返回值：0成功，-1失败
     */
    int APQ_HWPORT_get_dev_str_by_id(int device_type, int devid_type);

    /**
     * 根据设备类型和所属设备的权限类型获取权限字符名，用于打印结果
     *
     * @param device_type {@link  HwportDeviceTypeEnum}
     * @param perm_type   {@link  HwportStoragePermEnum} {@link  HwportNetPermEnum} {@link  HwportNormalPermEnum}
     * @return 返回值：0成功，-1失败
     */
    int APQ_HWPORT_get_perm_str_by_id(int device_type, int perm_type);

    /**
     * 定义可配置权限的设备类型枚举类型
     */
    enum HwportDeviceTypeEnum {
        APQ_HWPORT_DEVICE_TYPE_NORMAL(0),                // 普通类设备
        APQ_HWPORT_DEVICE_TYPE_STORAGE(1),                // 存储类设备
        APQ_HWPORT_DEVICE_TYPE_NET(2),                    // 网络类设备

        APQ_HWPORT_DEVICE_TYPE_MAX(-1),    /* 不使用 */;

        private int code;

        HwportDeviceTypeEnum(int code) {
            this.code = code;
        }
    }

    enum HwportResultEnum {
        APQ_HWPORT_RESULT_OK(0),                        // 调用成功
        APQ_HWPORT_RESULT_ERROR_UNKNOWN(1),            // 调用失败，未知原因
        APQ_HWPORT_RESULT_ERROR_INTERNAL(2),            // 调用失败，内部错误
        APQ_HWPORT_RESULT_ERROR_PARAM(3),                // 调用失败，参数错误
        APQ_HWPORT_RESULT_ERROR_RIGHTS(4),                // 调用失败，需要管理员权限

        ;

        private int code;

        HwportResultEnum(int code) {
            this.code = code;
        }
    }

    /**
     * 定义普通类设备ID（如USB设备）枚举类型
     */
    enum HwportNormalDevidEnum {
        APQ_HWPORT_NORMAL_DEVID_SERIAL(0),                // 串口
        APQ_HWPORT_NORMAL_DEVID_GPIO(1),                // GPIO
        APQ_HWPORT_NORMAL_DEVID_DONGLE(2),                // 加密狗
        APQ_HWPORT_NORMAL_DEVID_MOUSE(3),                // 鼠标
        APQ_HWPORT_NORMAL_DEVID_KEYBOARD(4),            // 键盘
        APQ_HWPORT_NORMAL_DEVID_CAMERA(5),                // 摄像头
        APQ_HWPORT_NORMAL_DEVID_USB_CONVERSION(6),        // USB转换
        APQ_HWPORT_NORMAL_DEVID_BT(7)                  // 蓝牙

        ;

        private int code;

        HwportNormalDevidEnum(int code) {
            this.code = code;
        }
    }

    /**
     * 定义普通类权限（如USB设备）枚举类型
     */
    enum HwportNormalPermEnum {
        APQ_HWPORT_NORMAL_PERM_UNKNOWN(0),                // 未知权限，一般是未配置
        APQ_HWPORT_NORMAL_PERM_DENY(1),                // 拒绝连接/使用此硬件接口
        APQ_HWPORT_NORMAL_PERM_ALLOW(2),                // 允许连接/使用此硬件接口
        APQ_HWPORT_NORMAL_PERM_MAX(-1);  /* 不使用 */

        private int code;

        HwportNormalPermEnum(int code) {
            this.code = code;
        }
    }

    /**
     * 定义存储类设备ID（存储设备）枚举类型
     */
    enum HwportStorageDevidEnum {
        APQ_HWPORT_STORAGE_DEVID_CDROM(0),                // 光驱
        APQ_HWPORT_STORAGE_DEVID_UDISK(1),                // U盘, removable disk
        //APQ_HWPORT_STORAGE_DEVID_HDD_DISK(2),			// 移动硬盘, <==> APQ_HWPORT_STORAGE_DEVID_UDISK
        //APQ_HWPORT_STORAGE_DEVID_USB_TYPEC(3),			// Type-C, <==> APQ_HWPORT_STORAGE_DEVID_UDISK

        APQ_HWPORT_STORAGE_DEVID_MAX(-1),    /* 不使用 */;

        private int code;

        HwportStorageDevidEnum(int code) {
            this.code = code;
        }
    }

    /**
     * 定义存储类权限（存储设备）枚举类型
     */
    enum HwportStoragePermEnum {
        APQ_HWPORT_STORAGE_PERM_UNKNOWN(0),            // 未知权限，一般是未配置
        APQ_HWPORT_STORAGE_PERM_DENY(1),                // 拒绝连接/使用此硬件接口
        APQ_HWPORT_STORAGE_PERM_ALLOW(2),                // 允许连接/使用此硬件接口
        APQ_HWPORT_STORAGE_PERM_READ_WRITE(3),            // 允许连接/使用此硬件接口，接口读写数据权限都有, <==> APQ_HWPORT_STORAGE_PERM_ALLOW
        APQ_HWPORT_STORAGE_PERM_READ_ONLY(4),            // 允许连接/使用此硬件接口，接口只有读数据权限 r
        APQ_HWPORT_STORAGE_PERM_WRITE_IN_ONLY(5),        // （等价于只读），允许连接/使用此硬件接口，接口只有拷入：支持读取数据以及向电脑拷入数据，不能从电脑拷出
        APQ_HWPORT_STORAGE_PERM_WRITE_OUT_ONLY(6),        // （此项无效，无法禁止向电脑拷入，等价于拒绝），允许连接/使用此硬件接口，接口只有拷出：支持读取数据以及从电脑拷出数据，不能向电脑拷入

        APQ_HWPORT_STORAGE_PERM_MAX(-1)    /* 不使用 */;

        private int code;

        HwportStoragePermEnum(int code) {
            this.code = code;
        }
    }

    /**
     * 定义网速类设备ID（网络设备）枚举类型
     */
    enum HwportNetDevidEnum {
        APQ_HWPORT_NET_DEVID_ETH_CARD(0),                // 网口
        //APQ_HWPORT_NET_DEVID_WIFI(1),					// WiFi <==> 网口
        //APQ_HWPORT_NET_DEVID_CELLULAR(2),				// 4G + 5G <==> 网口

        APQ_HWPORT_NET_DEVID_MAX(-1)    /* 不使用 */;

        private int code;

        HwportNetDevidEnum(int code) {
            this.code = code;
        }
    }

    /**
     * 定义网速类权限（网络设备）枚举类型
     */
    enum HwportNetPermEnum {
        APQ_HWPORT_NET_PERM_UNKNOWN(0),                // 未知权限，一般是未配置
        APQ_HWPORT_NET_PERM_DENY(1),                    // 拒绝连接/使用此硬件接口
        APQ_HWPORT_NET_PERM_ALLOW(2),                    // 允许连接/使用此硬件接口
        APQ_HWPORT_NET_PERM_NO_LIMIT(3),                // 不限制网络带宽, APQ_HWPORT_NET_PERM_ALLOW
        APQ_HWPORT_NET_PERM_LIMIT(4),                    // 限制网络带宽，速度值由参数speed指定

        APQ_HWPORT_NET_PERM_MAX(-1)    /* 不使用 */;

        private int code;

        HwportNetPermEnum(int code) {
            this.code = code;
        }
    }


}
