package com.priusis.controller;

import cn.hutool.core.io.unit.DataSizeUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.priusis.lib.R;
import com.priusis.service.common.MacAddrService;
import com.priusis.utils.CoreTempUtil;
import com.priusis.utils.Oshi4NoneSigarUtil;
import lombok.extern.slf4j.Slf4j;
import org.hyperic.sigar.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.PostConstruct;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author yangli
 * @since 2021/09/06
 */
@Slf4j
@RestController
@RequestMapping(value = "device_info")
public class DeviceInfoController {

    @Autowired
    private MacAddrService macAddrService;

    private Map<String, Object> info;

    private Locale enlocale  = new Locale("en", "US");
    private DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(enlocale);

    @GetMapping(value = "get")
    public R<Map<String, Object>> get() throws UnknownHostException, SocketException {
        if (info == null) refresh();
        return R.success(info);
    }

    @Scheduled(fixedDelay = 600000)
    protected void refresh() throws UnknownHostException {
        Sigar sigar = new Sigar();
        JSONObject diskModel = null;
        JSONObject diskCap = null;
        JSONObject diskFree = null;
        FileSystem[] diskStores;

        decimalFormat.applyPattern("#.00");

        try {
            String diskStoresFromOshiJson = Oshi4NoneSigarUtil.getDiskStoresModelJsonFromOshi();
            if (null != diskStoresFromOshiJson) {
                diskModel = JSONUtil.parseObj(diskStoresFromOshiJson);
            }

            diskStores = sigar.getFileSystemList();
            diskCap = JSONUtil.parseObj(Arrays.stream(diskStores)
                    .filter(partition -> StrUtil.isNotBlank(partition.getDevName()))
                    .filter(partition -> 2 == partition.getType())
                    .map(partition -> {
                        try {
                            FileSystemUsage usage = sigar.getFileSystemUsage(partition.getDirName());
                            return "\"" + StrUtil.removeSuffix(partition.getDevName(), ":\\") + "盘\":\"" +
                                    DataSizeUtil.format(usage.getTotal() * 1024) + "\"";
                        } catch (SigarException e) {
                            log.error("SigarException: {}", e.getMessage());
                        }
                        return "";
                    })
                    .collect(Collectors.joining(",", "{", "}")));
            diskFree = JSONUtil.parseObj(Arrays.stream(diskStores)
                    .filter(partition -> StrUtil.isNotBlank(partition.getDevName()))
                    .filter(partition -> 2 == partition.getType())
                    .map(partition -> {
                        try {
                            FileSystemUsage usage = sigar.getFileSystemUsage(partition.getDirName());
                            return "\"" + StrUtil.removeSuffix(partition.getDevName(), ":\\") + "盘\":\"" +
                                    decimalFormat.format(100d * usage.getFree() / usage.getTotal()) + "%\"";
                        } catch (SigarException e) {
                            log.error("SigarException: {}", e.getMessage());
                        }
                        return "";
                    })
                    .collect(Collectors.joining(",", "{", "}")));
        } catch (Exception e) {
            log.error("硬盘数据获取失败", e);
        }

        String memoryCap = "";
        String cpuModel = "";
//        double cpuLoad = 0;
//        String memoryAvailable = '';
        double cpuTemp = 0;
        String operatingSystem = "";
        String baseboard = "";
        String ip = "";

        try {
            baseboard = Oshi4NoneSigarUtil.getBaseboardWithCache();  // 获取主板信息  缓存
            ip = InetAddress.getLocalHost().getHostAddress();

            CpuInfo cpuInfo = sigar.getCpuInfoList()[0];
            cpuModel = cpuInfo.getVendor() + " " + cpuInfo.getModel();

            Mem mem = sigar.getMem();
            memoryCap = DataSizeUtil.format(mem.getTotal());

//            cpuLoad = Double.parseDouble(new DecimalFormat("#.00").format(sigar.getCpuPerc().getCombined() * 100));

//            memoryAvailable = new DecimalFormat("#,##0.###").format(mem.getFree() / Math.pow(1024, 3));

            OperatingSystem OS = OperatingSystem.getInstance();
            operatingSystem = OS.getVendor() + " " + OS.getName().replace("Win32", "Windows") + " " + OS.getVersion() + "(" + OS.getArch() + ")";

//            cpuTemp = Double.parseDouble(new DecimalFormat("#.00").format(OshiUtil.getSensors().getCpuTemperature()));
            cpuTemp = CoreTempUtil.getCpuTemp();
        } catch (Exception e) {
            log.error("SigarException", e);
        }


        info = MapUtil.<String, Object>builder()
                .put("operatingSystem", operatingSystem)
                .put("mac", macAddrService.getMacAddr())
                .put("baseboard", baseboard)
                .put("ip", ip)
                .put("cpuModel", cpuModel)
                .put("cpuTemp", cpuTemp)
//                .put("cpuLoad", cpuLoad)
//                .put("memoryAvailable", memoryAvailable)
                .put("memoryCap", memoryCap)
                .put("diskModel", diskModel)
                .put("diskCap", diskCap)
                .put("diskFree", diskFree)
                .build();
    }

    @PostConstruct
    public void init() {
        try {
            refresh();
        } catch (Exception e) {
            log.error("初始化失败", e);
        }
    }

    public static void main(String[] args) throws UnknownHostException, InterruptedException {
        DeviceInfoController deviceInfoController = new DeviceInfoController();
        while (true) {
            deviceInfoController.refresh();
            Thread.sleep(5000);
        }

    }
}
