package com.priusis.utils;

import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import oshi.util.FormatUtil;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;


/**
 * smartctl
 * https://www.smartmontools.org/
 */
@Slf4j
public class DiskSmartUtil {

    static final String SMARTCTL_EXE_PATH = ParseUrlUtil.parseAbsolutePath("/core/smartctl.exe", System.getProperty("user.dir") + "/apq-pc-client");

    /**
     * 获取硬盘温度smart信息
     *
     * @return
     * @throws IOException
     */
    public static Map<String, String> getDiskInfoFromSmart() throws IOException {
        Map<String, String> result = new HashMap<>();
        Process process = null;
        String smartScan = "";
        try {
            process = Runtime.getRuntime().exec(SMARTCTL_EXE_PATH + " --scan -j");  // AHCI ATA NVMe   https://blog.csdn.net/u012459957/article/details/51226658
            try (InputStream inputStream = process.getInputStream()) {
                smartScan = IOUtils.toString(inputStream, StandardCharsets.UTF_8);
            }

            int index = 0;
            JSONArray devices = JSONUtil.parseObj(smartScan).getJSONArray("devices");
            List<String> deviceDataEList = new ArrayList<>();
            for (int i = 0; i < devices.size(); i++) {
                String name = devices.getJSONObject(i).getStr("name");
                process = Runtime.getRuntime().exec(SMARTCTL_EXE_PATH + " -A -i " + name + " -j");
                String smartctlAinfo = "";
                try (InputStream inputStream = process.getInputStream()) {
                    smartctlAinfo = IOUtils.toString(inputStream, StandardCharsets.UTF_8);
                }
                JSONObject smartctlInfoJsonObject = JSONUtil.parseObj(smartctlAinfo);
                JSONObject temperature = smartctlInfoJsonObject.getJSONObject("temperature");
                String diskTemp = "?";
                if (null != temperature) {
                    diskTemp = temperature.getInt("current") + "";
                }
                String serialNumber = smartctlInfoJsonObject.getStr("serial_number");
                String modelName = smartctlInfoJsonObject.getStr("model_name");

                if (!deviceDataEList.contains(modelName)) {
                    StringBuffer sb = new StringBuffer();
                    String diskType = "";
                    if (smartctlAinfo.contains("HHD") || smartctlAinfo.contains("Head_Flying_Hours") || smartctlAinfo.contains("Load_Cycle_Count")) {
                        diskType = "机械硬盘";
                    } else if (smartctlAinfo.contains("SSD") || smartctlAinfo.contains("Solid State Device")) {
                        diskType = "固态硬盘";
                    }

                    sb.append("{");
                    sb.append("\"diskTemp\":\"").append(diskTemp).append("\",");
                    sb.append("\"diskType\":\"").append(diskType).append("\",");
                    sb.append("\"serialNumber\":\"").append(serialNumber).append("\"");
                    sb.append("}");
                    deviceDataEList.add(modelName);

                    result.put(modelName, sb.toString());
                }
            }

        } catch (Exception e) {
            log.error("获取硬盘smart信息异常", e);
        }

        if (null != process && process.isAlive()) {
            process.destroy();
        }
        return result;
    }

    public static String getDeviceTemp() {
        try {
            Map<String, String> diskInfoFromSmartMap = DiskSmartUtil.getDiskInfoFromSmart();
            StringBuilder sb = new StringBuilder();
            for (String key : diskInfoFromSmartMap.keySet()) {
                JSONObject diskInfoFromJson = JSONUtil.parseObj(diskInfoFromSmartMap.get(key));
                String diskTemp = diskInfoFromJson.getStr("diskTemp");
                if (!StrUtil.equals("?", diskTemp)) {
                    sb.append(key).append(":").append(diskTemp).append("℃");
                    sb.append(",");
                }
            }

            return sb.substring(0, sb.length() - 1);
        } catch (IOException e) {
            log.error("获取硬盘温度smart信息异常", e);
        }

        return "";
    }

    public static void main(String[] args) throws IOException {
        Map<String, String> diskInfoFromSmartMap = DiskSmartUtil.getDiskInfoFromSmart();
        System.out.println(diskInfoFromSmartMap);

        StringBuilder sb = new StringBuilder();
        for (String key : diskInfoFromSmartMap.keySet()) {
            JSONObject diskInfoFromJson = JSONUtil.parseObj(diskInfoFromSmartMap.get(key));
            sb.append(key).append(":").append(diskInfoFromJson.getStr("diskTemp")).append("℃");
            sb.append(",");
        }

        System.out.println(sb.substring(0, sb.length() - 1).toString());
    }

}
