package com.priusis.controller;

import cn.hutool.core.io.unit.DataSizeUtil;
import cn.hutool.core.map.MapUtil;
import com.priusis.lib.R;
import com.priusis.service.common.MacAddrService;
import com.priusis.utils.Oshi4NoneSigarUtil;
import lombok.extern.slf4j.Slf4j;
import org.hyperic.sigar.CpuInfo;
import org.hyperic.sigar.Mem;
import org.hyperic.sigar.OperatingSystem;
import org.hyperic.sigar.Sigar;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.PostConstruct;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Locale;
import java.util.Map;

/**
 * @author yangli
 * @since 2021/09/06
 */
@Slf4j
@RestController
@RequestMapping(value = "device_info")
public class DeviceInfoController {

    @Autowired
    private MacAddrService macAddrService;

    private Map<String, Object> info;

    private Locale enlocale  = new Locale("en", "US");
    private DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(enlocale);

    @GetMapping(value = "get")
    public R<Map<String, Object>> get() throws UnknownHostException, SocketException {
        if (info == null) refresh();
        return R.success(info);
    }

    @Scheduled(fixedDelay = 600000)
    protected void refresh() throws UnknownHostException {
        Sigar sigar = new Sigar();

        decimalFormat.applyPattern("#.00");

        String memoryCap = "";
        String cpuModel = "";
        double cpuTemp = 0;
        String operatingSystem = "";
        String baseboard = "";
        String ip = "";

        try {
            baseboard = Oshi4NoneSigarUtil.getBaseboardWithCache();  // 获取主板信息  缓存
            ip = InetAddress.getLocalHost().getHostAddress();

            CpuInfo cpuInfo = sigar.getCpuInfoList()[0];
            cpuModel = cpuInfo.getVendor() + " " + cpuInfo.getModel();

            Mem mem = sigar.getMem();
            memoryCap = DataSizeUtil.format(mem.getTotal());

            OperatingSystem OS = OperatingSystem.getInstance();
            operatingSystem = OS.getVendor() + " " + OS.getName().replace("Win32", "Windows") + " " + OS.getVersion() + "(" + OS.getArch() + ")";

        } catch (Exception e) {
            log.error("SigarException", e);
        }


        info = MapUtil.<String, Object>builder()
                .put("operatingSystem", operatingSystem)
                .put("mac", macAddrService.getMacAddr())
                .put("baseboard", baseboard)
                .put("ip", ip)
                .put("cpuModel", cpuModel)
                .put("cpuTemp", cpuTemp)
                .put("memoryCap", memoryCap)
                .build();
    }

    @PostConstruct
    public void init() {
        try {
            refresh();
        } catch (Exception e) {
            log.error("初始化失败", e);
        }
    }

    public static void main(String[] args) throws UnknownHostException, InterruptedException {
        DeviceInfoController deviceInfoController = new DeviceInfoController();
        while (true) {
            deviceInfoController.refresh();
            Thread.sleep(5000);
        }

    }
}
