package com.priusis.controller;

import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONUtil;
import com.priusis.client.service.MqttRpcDataMessage;
import com.priusis.client.service.TenantManagerService;
import com.priusis.lib.R;
import com.priusis.util.SysConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.IOException;
import java.util.Map;

/**
 * @author yangli
 * @since 2021/09/10
 */
@Slf4j
@Validated
@RestController
@RequestMapping(value = "config")
public class ConfigController {

    @Autowired
    private TenantManagerService service;

    @GetMapping(value = "get_download_path")
    public R<String> getDownloadPath() throws IOException {
        String path = SysConfigUtil.getProperty("download.path",
                System.getProperty("user.home") + File.separator + "Downloads");
        return R.success(path);
    }

    @PostMapping(value = "save_download_path")
    public R<Void> saveDownloadPath(@RequestBody Map<String, String> params) throws IOException {
        String path = params.get("path");
        SysConfigUtil.saveProperty("download.path", path);
        return R.success();
    }

    @GetMapping(value = "get_device_sn")
    public R<String> getDeviceSN() throws IOException {
        return R.success(SysConfigUtil.getProperty("device.sn"));
    }

    @PostMapping(value = "save_device_sn")
    public R<Void> saveDeviceSN(@RequestBody Map<String, String> params) throws IOException {
        String sn = params.get("sn");
        SysConfigUtil.saveProperty("device.sn", sn);
        return R.success();
    }

    @GetMapping(value = "get_data_reporting")
    public R<JSONArray> getDataReporting() throws IOException {
        return R.success(JSONUtil.parseArray(SysConfigUtil.getProperty("data.reporting",
                "[" +
                        "{\"code\":\"cpuLoad\",\"name\":\"CPU负载\",\"frequency\":30}," +
                        "{\"code\":\"cpuTemp\",\"name\":\"CPU温度\",\"frequency\":30}," +
                        "{\"code\":\"mac\",\"name\":\"MAC地址\",\"frequency\":30}," +
                        "{\"code\":\"memoryCap\",\"name\":\"内存容量\",\"frequency\":30}," +
                        "{\"code\":\"diskCap\",\"name\":\"硬盘容量\",\"frequency\":30}," +
                        "{\"code\":\"diskModel\",\"name\":\"硬盘型号\",\"frequency\":60}" +
                        "]")));
    }

    @PostMapping(value = "save_data_reporting")
    public R<Void> saveDataReporting(@RequestBody JSONArray array) throws IOException {
        String dataReport = array.toString();
        SysConfigUtil.saveProperty("data.reporting", dataReport);
        // 存储配置作为Rpc下发的数据
        try {
            MqttRpcDataMessage mqttRpcDataMessage = MqttRpcDataMessage.builder()
                    .method("info").params(dataReport).build();
            service.flushRpcDataToFile(mqttRpcDataMessage);
        } catch (IOException e) {
            log.error("Failed to save_data_reporting persistent : {}", dataReport);
        }
        return R.success();
    }


    @PostConstruct
    public void init() {
        try {
            // 初始化自动保存一次数据上报设置
            saveDataReporting(getDataReporting().getData());
        } catch (Exception e) {
            log.error("初始化异常", e);
        }
    }
}
