package com.priusis.controller;

import cn.hutool.core.io.unit.DataSizeUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import cn.hutool.system.oshi.OshiUtil;
import com.priusis.lib.R;
import com.priusis.util.MacAddrUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import oshi.hardware.HWDiskStore;
import oshi.hardware.HWPartition;

import javax.annotation.PostConstruct;
import java.io.File;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author yangli
 * @since 2021/09/06
 */
@Slf4j
@RestController
@RequestMapping(value = "device_info")
public class DeviceInfoController {

    private Map<String, Object> info;

    @GetMapping(value = "get")
    public R<Map<String, Object>> get() throws UnknownHostException, SocketException {
        if (info == null) refresh();
        return R.success(info);
    }

    @Scheduled(fixedDelay = 5000)
    protected void refresh() throws UnknownHostException {
        JSONObject diskModel = null;
        JSONObject diskCap = null;
        JSONObject diskFree = null;
        try {
            diskModel = JSONUtil.parseObj(Arrays.stream(OshiUtil.getHardware().getDiskStores())
                    .map(store -> Arrays.stream(store.getPartitions())
                            .map(HWPartition::getMountPoint)
                            .filter(StrUtil::isNotBlank)
                            .map(s -> "\"" + StrUtil.removeSuffix(s, ":\\") + "盘\":\"" +
                                    StrUtil.removeSuffix(store.getModel(), " (标准磁盘驱动器)") + "\"")
                            .collect(Collectors.joining(",")))
                    .filter(StrUtil::isNotBlank)
                    .collect(Collectors.joining(",", "{", "}")));
            diskCap = JSONUtil.parseObj(Arrays.stream(OshiUtil.getHardware().getDiskStores())
                    .map(HWDiskStore::getPartitions)
                    .flatMap(Arrays::stream)
                    .filter(partition -> StrUtil.isNotBlank(partition.getMountPoint()))
                    .map(partition -> "\"" + StrUtil.removeSuffix(partition.getMountPoint(), ":\\") + "盘\":\"" +
                            DataSizeUtil.format(partition.getSize()) + "\"")
                    .collect(Collectors.joining(",", "{", "}")));
            diskFree = JSONUtil.parseObj(Arrays.stream(OshiUtil.getHardware().getDiskStores())
                    .map(HWDiskStore::getPartitions)
                    .flatMap(Arrays::stream)
                    .filter(partition -> StrUtil.isNotBlank(partition.getMountPoint()))
                    .map(partition -> "\"" + StrUtil.removeSuffix(partition.getMountPoint(), ":\\") + "盘\":\"" +
                            new DecimalFormat("#.00").format(100d * new File(partition.getMountPoint()).getFreeSpace() / partition.getSize()) + "%\"")
                    .collect(Collectors.joining(",", "{", "}")));
        } catch (Exception e) {
            log.error("硬盘数据获取失败", e);
        }
        info = MapUtil.<String, Object>builder()
                .put("operatingSystem", OshiUtil.getOs().toString())
                .put("mac", MacAddrUtil.get())
                .put("baseboard", OshiUtil.getSystem().getBaseboard().getManufacturer() + " " + OshiUtil.getSystem().getBaseboard().getVersion())
                .put("ip", InetAddress.getLocalHost().getHostAddress())
                .put("cpuModel", OshiUtil.getCpuInfo(0).getCpuModel())
                .put("cpuTemp", Double.parseDouble(new DecimalFormat("#.00").format(OshiUtil.getSensors().getCpuTemperature())))
                .put("memoryCap", DataSizeUtil.format(OshiUtil.getMemory().getTotal()))
                .put("diskModel", diskModel)
                .put("diskCap", diskCap)
                .put("diskFree", diskFree)
                .build();
    }

    @PostConstruct
    public void init() {
        try {
            refresh();
        } catch (Exception e) {
            log.error("初始化失败", e);
        }
    }

    public static void main(String[] args) throws UnknownHostException {
        System.out.println("操作系统");
        System.out.println(OshiUtil.getOs().toString());
        System.out.println("MAC地址");
        System.out.println(MacAddrUtil.get());
        System.out.println("主板");
        System.out.println(OshiUtil.getSystem().getBaseboard().getManufacturer() + " " + OshiUtil.getSystem().getBaseboard().getVersion());
        System.out.println("IP");
        System.out.println(InetAddress.getLocalHost().getHostAddress());
        System.out.println("CPU型号");
        System.out.println(OshiUtil.getCpuInfo(0).getCpuModel());
        System.out.println("CPU温度");
        System.out.println(Double.parseDouble(new DecimalFormat("#.00").format(OshiUtil.getSensors().getCpuTemperature())));
        System.out.println("内存容量");
        System.out.println(DataSizeUtil.format(OshiUtil.getMemory().getTotal()));
        System.out.println("硬盘型号");
        System.out.println(Arrays.stream(OshiUtil.getHardware().getDiskStores())
                .map(store -> Arrays.stream(store.getPartitions())
                        .map(HWPartition::getMountPoint)
                        .filter(StrUtil::isNotBlank)
                        .map(s -> "\"" + StrUtil.removeSuffix(s, ":\\") + "盘\":\"" +
                                StrUtil.removeSuffix(store.getModel(), " (标准磁盘驱动器)") + "\"")
                        .collect(Collectors.joining(",")))
                .filter(StrUtil::isNotBlank)
                .collect(Collectors.joining(",", "{", "}")));
        System.out.println("硬盘容量");
        System.out.println(Arrays.stream(OshiUtil.getHardware().getDiskStores())
                .map(HWDiskStore::getPartitions)
                .flatMap(Arrays::stream)
                .filter(partition -> StrUtil.isNotBlank(partition.getMountPoint()))
                .map(partition -> "\"" + StrUtil.removeSuffix(partition.getMountPoint(), ":\\") + "盘\":\"" +
                        DataSizeUtil.format(partition.getSize()) + "\"")
                .collect(Collectors.joining(",", "{", "}")));
        System.out.println("硬盘剩余");
        DecimalFormat formatter = new DecimalFormat("#.00");
        System.out.println(Arrays.stream(OshiUtil.getHardware().getDiskStores())
                .map(HWDiskStore::getPartitions)
                .flatMap(Arrays::stream)
                .filter(partition -> StrUtil.isNotBlank(partition.getMountPoint()))
                .map(partition -> "\"" + StrUtil.removeSuffix(partition.getMountPoint(), ":\\") + "盘\":\"" +
                        formatter.format(100d * new File(partition.getMountPoint()).getFreeSpace() / partition.getSize()) + "%\"")
                .collect(Collectors.joining(",", "{", "}")));

    }
}
