package com.priusis.job;

import cn.hutool.core.io.unit.DataSizeUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.priusis.client.service.MqttRpcDataMessage;
import com.priusis.service.common.MacAddrService;
import com.priusis.utils.Oshi4NoneSigarUtil;
import com.priusis.utils.sigar.SigarUtil;
import com.priusis.vo.ApqInfoDataVo;
import lombok.extern.slf4j.Slf4j;
import org.hyperic.sigar.Mem;
import org.hyperic.sigar.Sigar;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.net.InetAddress;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

@Slf4j
@Component
@ConditionalOnExpression("'${apq.job.type:null}'=='all' || '${apq.job.type:null}'=='info'")
public class ApqInfoJob {

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private MacAddrService macAddrService;

    @Value(value = "${apq.iot-gateway}")
    private String gateway;

    @Value(value = "${apq.client.tenantId}")
    private Long tenantId;

    @Value(value = "${apq.client.productId}")
    private Long productId;

    @Value(value = "${apq.url.device_info}")
    private String deviceUpdateNameUrl;

    private int runningIndex = 0;


    private Locale enlocale  = new Locale("en", "US");
    private DecimalFormat decimalFormat = (DecimalFormat)NumberFormat.getNumberInstance(enlocale);

    // 30秒一次上报
    @Scheduled(fixedDelay = 30000L)
    protected void controlProgramTask() {
        decimalFormat.applyPattern("#.00");
        log.info("设备实时数据采集上报=================");
        runningIndex++;
        boolean isNoneEven = runningIndex % 2 == 0;

        // 获取采集配置，数据上报频率 params == data
        ResponseEntity<MqttRpcDataMessage> forEntity = null;
        Map<String, Integer> mapParams = null;
        try {
            forEntity = restTemplate.getForEntity("http://localhost:8765/rpc_cmd/info", MqttRpcDataMessage.class);
            if (null != forEntity) {
                MqttRpcDataMessage body = forEntity.getBody();
                String params = body.getParams();

                if (StrUtil.isNotBlank(params)) {
                    List<ApqInfoDataVo> apqInfoDataVos = JSONUtil.toList(params, ApqInfoDataVo.class);
                    mapParams = new HashMap<>();
                    for (ApqInfoDataVo apqInfoDataVo : apqInfoDataVos) {
                        mapParams.put(apqInfoDataVo.getCode(), apqInfoDataVo.getFrequency());
                    }
                }
            }
        } catch (Exception e) {
            log.error("采集数据异常", e);
            return;
        }


        String operatingSystem = "operatingSystem";
        String mac = "mac";
        String baseboard = "baseboard";
        String ip = "ip";
        String cpuModel = "cpuModel";
        String cpuLoad = "cpuLoad";
        String memoryCap = "memoryCap";
        String memoryAvailable = "memoryAvailable";
        String memoryOccupyRate = "memoryOccupyRate";
        String ethernet = "ethernet";

        Map<String, Object> data = null;
        try {
            Sigar sigar = new Sigar();
            String memoryCapV = "";

            String cpuModelV = "";
            double cpuLoadV = 0;
            String memoryAvailableV = "";
            double memoryOccupyRateV = 0;

            String operatingSystemV = "";
            String baseboardV = "";
            String ipV = "";
            String ethernetV = "";

            try {
                baseboardV = Oshi4NoneSigarUtil.getBaseboardWithCache();  // 获取主板信息  缓存;
                ipV = InetAddress.getLocalHost().getHostAddress();
                ethernetV = SigarUtil.ethernetSimple(sigar);

                org.hyperic.sigar.CpuInfo cpuInfo = sigar.getCpuInfoList()[0];
                cpuModelV = cpuInfo.getVendor() + " " + cpuInfo.getModel();

                cpuLoadV = Double.parseDouble(decimalFormat.format(sigar.getCpuPerc().getCombined() * 100));
            } catch (Exception e) {
                log.error("SigarException", e);
            }

            try {
                Mem mem = sigar.getMem();
                memoryCapV = DataSizeUtil.format(mem.getTotal());

                log.info("===============mem:{}", mem.getUsed() * 1.0 / mem.getTotal() * 100);

                memoryOccupyRateV = Double.parseDouble(decimalFormat.format(mem.getUsed() * 1.0 / mem.getTotal() * 100));

                memoryAvailableV = DataSizeUtil.format(mem.getUsed()) + "/" + DataSizeUtil.format(mem.getTotal())
                        + "(" + memoryOccupyRateV + "%)";

                operatingSystemV = Oshi4NoneSigarUtil.getOsWithCache();

            } catch (Exception e) {
                log.error("SigarException", e);
            }

            data = MapUtil.<String, Object>builder()
                    .put(isNeedCollection(isNoneEven, operatingSystem, mapParams), operatingSystem, operatingSystemV)
                    .put(isNeedCollection(isNoneEven, mac, mapParams), mac, macAddrService.getMacAddr())
                    .put(isNeedCollection(isNoneEven, baseboard, mapParams), baseboard, baseboardV)
                    .put(isNeedCollection(isNoneEven, ip, mapParams), ip, ipV)
                    .put(isNeedCollection(isNoneEven, cpuModel, mapParams), cpuModel, cpuModelV)
                    .put(isNeedCollection(isNoneEven, cpuLoad, mapParams), cpuLoad, cpuLoadV)
                    .put(isNeedCollection(isNoneEven, memoryCap, mapParams), memoryCap, memoryCapV)
                    .put(isNeedCollection(isNoneEven, memoryAvailable, mapParams), memoryAvailable, memoryAvailableV)
                    .put(isNeedCollection(isNoneEven, memoryOccupyRate, mapParams), memoryOccupyRate, memoryOccupyRateV)
                    .put(isNeedCollection(isNoneEven, ethernet, mapParams), ethernet, ethernetV)
                    .build();
            log.info("采集数据，上报属性: mapData:{}", data);
            Map mapR = restTemplate.postForObject("http://localhost:8765/uplink/oc-client", data, Map.class);
            log.info("采集数据，上报属性: ret:{}", mapR);
        } catch (Exception e) {
            log.error("采集数据异常", e);
        }
    }



    private boolean isNeedCollection(boolean isNoneEven, String key, Map<String, Integer> mapParams) {
        if (null != mapParams && mapParams.containsKey(key)) {
            return !isNoneEven || mapParams.get(key).equals(30);
        }
        return isNoneEven;
    }
}
