package com.priusis.job;

import cn.hutool.core.io.unit.DataSizeUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONUtil;
import cn.hutool.system.oshi.CpuInfo;
import cn.hutool.system.oshi.OshiUtil;
import com.priusis.service.common.MacAddrService;
import com.priusis.utils.CoreTempUtil;
import com.priusis.utils.DiskSmartUtil;
import com.priusis.utils.Oshi4NoneSigarUtil;
import com.priusis.utils.sigar.SigarUtil;
import com.priusis.vo.ApqInfoDataVo;
import com.priusis.client.service.MqttRpcDataMessage;
import com.priusis.vo.RemoteDeviceVoResult;
import lombok.extern.slf4j.Slf4j;
import org.hyperic.sigar.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import oshi.hardware.HWDiskStore;
import oshi.hardware.HWPartition;
import oshi.util.FormatUtil;

import java.net.InetAddress;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Component
@ConditionalOnExpression("'${apq.job.type:null}'=='all' || '${apq.job.type:null}'=='info'")
public class ApqInfoJob {

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private MacAddrService macAddrService;

    @Value(value = "${apq.iot-gateway}")
    private String gateway;

    @Value(value = "${apq.client.tenantId}")
    private Long tenantId;

    @Value(value = "${apq.client.productId}")
    private Long productId;

    @Value(value = "${apq.url.device_info}")
    private String deviceUpdateNameUrl;

    private int runningIndex = 0;

    private static Map<String, Map<String, String>> hardwareCache = new HashMap<>();
    private static Map<String, String> attrCache = new HashMap<>();
    private Locale enlocale  = new Locale("en", "US");
    private DecimalFormat decimalFormat = (DecimalFormat)NumberFormat.getNumberInstance(enlocale);

    // 30秒一次上报
    @Scheduled(fixedDelay = 30000L)
    protected void controlProgramTask() {
        decimalFormat.applyPattern("#.00");
        log.info("设备实时数据采集上报=================");
        runningIndex++;
        boolean isNoneEven = runningIndex % 2 == 0;

        // 获取采集配置，数据上报频率 params == data
        ResponseEntity<MqttRpcDataMessage> forEntity = null;
        Map<String, Integer> mapParams = null;
        try {
            forEntity = restTemplate.getForEntity("http://localhost:8765/rpc_cmd/info", MqttRpcDataMessage.class);
            if (null != forEntity) {
                MqttRpcDataMessage body = forEntity.getBody();
                String params = body.getParams();

                if (StrUtil.isNotBlank(params)) {
                    List<ApqInfoDataVo> apqInfoDataVos = JSONUtil.toList(params, ApqInfoDataVo.class);
                    mapParams = new HashMap<>();
                    for (ApqInfoDataVo apqInfoDataVo : apqInfoDataVos) {
                        mapParams.put(apqInfoDataVo.getCode(), apqInfoDataVo.getFrequency());
                    }
                }
            }
        } catch (Exception e) {
            log.error("采集数据异常", e);
            return;
        }


        String operatingSystem = "operatingSystem";
        String mac = "mac";
        String baseboard = "baseboard";
        String ip = "ip";
        String cpuModel = "cpuModel";
        String cpuLoad = "cpuLoad";
        String cpuTemp = "cpuTemp";
        String gpuTemp = "gpuTemp";
        String memoryCap = "memoryCap";
        String memoryAvailable = "memoryAvailable";
        String memoryOccupyRate = "memoryOccupyRate";
        String diskSpeed = "diskSpeed";
        String diskModel = "diskModel";
        String diskCap = "diskCap";
        String diskFree = "diskFree";
        String diskTemp = "diskTemp";
        String ethernet = "ethernet";

        Map<String, Object> data = null;
        try {
            Sigar sigar = new Sigar();
            // 上报PC实时信息
            String collect = Oshi4NoneSigarUtil.getDiskStoresModelFromOshi();
            String collectForCheck = Oshi4NoneSigarUtil.getDiskStoresModelJsonFromOshi();
            // 硬盘型号/ID变化后报警
            Map<String, String> hardwareCacheRefresh = JSONUtil.toBean(collectForCheck, Map.class);
            checkDiskHardware("hard_disk", hardwareCacheRefresh);

            // 设备名称变更
            checkFacilityName();

            String collect1 = null;
            String collect2 = null;
            FileSystem[] diskStores;
            try {
                diskStores = sigar.getFileSystemList();
                collect1 = Arrays.stream(diskStores)
                        .filter(partition -> StrUtil.isNotBlank(partition.getDevName()))
                        .filter(partition -> 2 == partition.getType())
                        .map(partition -> {
                            try {
                                FileSystemUsage usage = sigar.getFileSystemUsage(partition.getDirName());
                                return "\"" + StrUtil.removeSuffix(partition.getDevName(), ":\\") + "盘\":\"" +
                                        DataSizeUtil.format(usage.getTotal() * 1024) + "\"";
                            } catch (SigarException e) {
                                log.error("SigarException: {}", e.getMessage());
                            }
                            return "";
                        })
                        .collect(Collectors.joining(",", "{", "}"));
                collect2 = Arrays.stream(diskStores)
                        .filter(partition -> StrUtil.isNotBlank(partition.getDevName()))
                        .filter(partition -> 2 == partition.getType())
                        .map(partition -> {
                            try {
                                FileSystemUsage usage = sigar.getFileSystemUsage(partition.getDirName());
                                return "\"" + StrUtil.removeSuffix(partition.getDevName(), ":\\") + "盘\":\"" +
                                        decimalFormat.format(100d * (usage.getTotal() - usage.getFree()) / usage.getTotal()) + "%\"";
                            } catch (SigarException e) {
                                log.error("SigarException: {}", e.getMessage());
                            }
                            return "";
                        })
                        .collect(Collectors.joining(",", "{", "}"));
            } catch (Exception e) {
                log.error("硬盘数据获取失败", e);
            }

            String memoryCapV = "";
            String diskSpeedV = "";
            String diskTempV = "";  // 获取硬盘温度smart信息
            String cpuModelV = "";
            double cpuLoadV = 0;
            String memoryAvailableV = "";
            double memoryOccupyRateV = 0;
            double cpuTempV = 0;
            double gpuTempV = 0;
            String operatingSystemV = "";
            String baseboardV = "";
            String ipV = "";
            String ethernetV = "";

            try {
                baseboardV = Oshi4NoneSigarUtil.getBaseboardWithCache();  // 获取主板信息  缓存;
                ipV = InetAddress.getLocalHost().getHostAddress();
                ethernetV = SigarUtil.ethernetSimple(sigar);
                diskTempV = DiskSmartUtil.getDeviceTemp();

                org.hyperic.sigar.CpuInfo cpuInfo = sigar.getCpuInfoList()[0];
                cpuModelV = cpuInfo.getVendor() + " " + cpuInfo.getModel();

                cpuLoadV = Double.parseDouble(decimalFormat.format(sigar.getCpuPerc().getCombined() * 100));
            } catch (Exception e) {
                log.error("SigarException", e);
            }

            try {
                Mem mem = sigar.getMem();
                memoryCapV = DataSizeUtil.format(mem.getTotal());

                log.info("===============mem:{}", mem.getUsed() * 1.0 / mem.getTotal() * 100);

                memoryOccupyRateV = Double.parseDouble(decimalFormat.format(mem.getUsed() * 1.0 / mem.getTotal() * 100));

                memoryAvailableV = DataSizeUtil.format(mem.getUsed()) + "/" + DataSizeUtil.format(mem.getTotal())
                        + "(" + memoryOccupyRateV + "%)";

//                OperatingSystem OS = OperatingSystem.getInstance();
                operatingSystemV = Oshi4NoneSigarUtil.getOsWithCache();

                // 用WMI接口读出的是CPU温度 getCpuTemperature   == MSAcpiThermalZoneTemperature.queryCurrentTemperature();
                // WMI接口读出的就是这个ACPI Thermal Zone的温度，是主板的温区温度，而不是CPU的温度
                // 1. WMI需要BIOS支持，而很多BIOS并不支持
                // 2. Thermal Zone的_TMP报告的一般是主板的某个温区，而不是CPU的温度

                // 最佳实践 https://www.cnblogs.com/javawebsoa/archive/2013/05/31/3111351.html  WinRing0.sys + rdmsr指令
//                cpuTempV = Double.parseDouble(decimalFormat.format(OshiUtil.getSensors().getCpuTemperature()));
                int[] cpuAndGpuTemp = CoreTempUtil.getCpuAndGpuTemp();
                cpuTempV = cpuAndGpuTemp[0];
                gpuTempV = cpuAndGpuTemp[1];
            } catch (Exception e) {
                log.error("SigarException", e);
            }

            diskSpeedV = SigarUtil.diskSpeed(sigar);
            data = MapUtil.<String, Object>builder()
                    .put(isNeedCollection(isNoneEven, operatingSystem, mapParams), operatingSystem, operatingSystemV)
                    .put(isNeedCollection(isNoneEven, mac, mapParams), mac, macAddrService.getMacAddr())
                    .put(isNeedCollection(isNoneEven, baseboard, mapParams), baseboard, baseboardV)
                    .put(isNeedCollection(isNoneEven, ip, mapParams), ip, ipV)
                    .put(isNeedCollection(isNoneEven, cpuModel, mapParams), cpuModel, cpuModelV)
                    .put(isNeedCollection(isNoneEven, cpuTemp, mapParams), cpuTemp, cpuTempV)
                    .put(isNeedCollection(isNoneEven, cpuLoad, mapParams), cpuLoad, cpuLoadV)
                    .put(isNeedCollection(isNoneEven, gpuTemp, mapParams), gpuTemp, gpuTempV)
                    .put(isNeedCollection(isNoneEven, memoryCap, mapParams), memoryCap, memoryCapV)
                    .put(isNeedCollection(isNoneEven, memoryAvailable, mapParams), memoryAvailable, memoryAvailableV)
                    .put(isNeedCollection(isNoneEven, memoryOccupyRate, mapParams), memoryOccupyRate, memoryOccupyRateV)
                    .put(isNeedCollection(isNoneEven, diskSpeed, mapParams), diskSpeed, diskSpeedV)
                    .put(isNeedCollection(isNoneEven, diskTemp, mapParams), diskTemp, diskTempV)
                    .put(isNeedCollection(isNoneEven, diskModel, mapParams), diskModel, collect)
                    .put(isNeedCollection(isNoneEven, diskCap, mapParams), diskCap, collect1)
                    .put(isNeedCollection(isNoneEven, diskFree, mapParams), diskFree, collect2)
                    .put(isNeedCollection(isNoneEven, ethernet, mapParams), ethernet, ethernetV)
                    .build();
            log.info("采集数据，上报属性: mapData:{}", data);
            Map mapR = restTemplate.postForObject("http://localhost:8765/uplink/oc-client", data, Map.class);
            log.info("采集数据，上报属性: ret:{}", mapR);
        } catch (Exception e) {
            log.error("采集数据异常", e);
            data = MapUtil.<String, Object>builder().put("ERROR", e.getMessage()).build();
        }
    }

    private void checkFacilityName() {
        Map<String, String> map = System.getenv();
        String computerName = map.get("COMPUTERNAME");// 获取计算机名
        if (attrCache.isEmpty() || !StrUtil.equals(attrCache.get("FacilityName"), computerName)) {
            Map<String, Object> params = new HashMap<>();
            params.put("tenantId", tenantId);
            params.put("productId", productId);
            params.put("macAddress", macAddrService.getMacAddr());
            params.put("name", computerName);
            String remoteDeviceVoJson = HttpUtil.post(gateway + deviceUpdateNameUrl, JSONUtil.toJsonStr(params), 5000);
            RemoteDeviceVoResult remoteDeviceVo = JSONUtil.toBean(remoteDeviceVoJson, RemoteDeviceVoResult.class);
            if (remoteDeviceVo.getCode() == 0) {
                attrCache.put("FacilityName", computerName);
            }
        }
    }

    /**
     * 硬盘型号/ID变化后报警
     * + 其他
     *
     * @param type                 类型
     * @param hardwareCacheRefresh
     */
    private void checkDiskHardware(String type, Map<String, String> hardwareCacheRefresh) {
        try {
            if (hardwareCache.isEmpty()) {
                // 获取采集到的硬件信息
                ResponseEntity<MqttRpcDataMessage> forEntity = null;
                try {
                    forEntity = restTemplate.getForEntity("http://localhost:8765/rpc_cmd/hardware_warning", MqttRpcDataMessage.class);
                    if (null != forEntity) {
                        MqttRpcDataMessage body = forEntity.getBody();
                        String params = body.getParams();

                        if (StrUtil.isNotBlank(params)) {
                            Map<String, Map<String, String>> hardwareCacheFromPersistence = JSONUtil.toBean(params, Map.class);
                            hardwareCache.putAll(hardwareCacheFromPersistence);
                        }
                    }
                } catch (Exception e) {
                    log.error("采集数据异常", e);
                }
            }
            Map<String, String> hardwareCache4ThisTypeMap = hardwareCache.get(type);
            if (null != hardwareCache4ThisTypeMap && !hardwareCache4ThisTypeMap.isEmpty()) {
                // 上报硬件监控数据
                Map mapRequest = new HashMap();
                hardwareCacheRefresh.forEach((k, v) -> {
                    if (!hardwareCache4ThisTypeMap.containsValue(v)) {
                        Map mapRequestDetail = new HashMap();
                        mapRequestDetail.put("name", k);
                        mapRequestDetail.put("value", v);
                        mapRequestDetail.put("type", "Add");
                        mapRequest.put(k, mapRequestDetail);
                    }
                });

                hardwareCache4ThisTypeMap.forEach((k, v) -> {
                    if (!hardwareCacheRefresh.containsValue(v)) {
                        Map mapRequestDetail = new HashMap();
                        mapRequestDetail.put("name", k);
                        mapRequestDetail.put("value", v);
                        mapRequestDetail.put("type", "Reduce");
                        mapRequest.put(k, mapRequestDetail);
                    }
                });

                if (!mapRequest.isEmpty()) {
                    mapRequest.put("type", type);
                    Map eventMapRequest = new HashMap();
                    eventMapRequest.put("requestId", 5);
                    eventMapRequest.put("methodName", "hardware_warning");
                    eventMapRequest.put("params", mapRequest);
                    Map map = restTemplate.postForObject("http://localhost:8765/uplink_event/oc-client", eventMapRequest, Map.class);
                    log.info("存在硬件扫描告警，上报告警事件: mapData:{}, ret:{}", mapRequest, map);
                    hardwareCache.put(type, hardwareCacheRefresh);

                    // 更新采集到的硬件信息
                    try {
                        MqttRpcDataMessage mqttRpcDataMessage = MqttRpcDataMessage.builder()
                                .sendTime(System.currentTimeMillis())
                                .method("hardware_warning")
                                .params(JSONUtil.toJsonStr(hardwareCache)).build();

                        restTemplate.postForObject("http://localhost:8765/rpc_cmd", mqttRpcDataMessage, String.class);
                    } catch (Exception e) {
                        log.error("采集数据异常", e);
                    }
                }
            } else {
                hardwareCache.put(type, hardwareCacheRefresh);
                // 更新采集到的硬件信息
                try {
                    MqttRpcDataMessage mqttRpcDataMessage = MqttRpcDataMessage.builder()
                            .sendTime(System.currentTimeMillis())
                            .method("hardware_warning")
                            .params(JSONUtil.toJsonStr(hardwareCache)).build();
                    restTemplate.postForObject("http://localhost:8765/rpc_cmd", mqttRpcDataMessage, String.class);
                } catch (Exception e) {
                    log.error("采集数据异常", e);
                }
            }
        } catch (RestClientException e) {
            log.error("硬盘型号/ID变化后报警异常");
        }
    }

    private boolean isNeedCollection(boolean isNoneEven, String key, Map<String, Integer> mapParams) {
        if (null != mapParams && mapParams.containsKey(key)) {
            return !isNoneEven || mapParams.get(key).equals(30);
        }
        return isNoneEven;
    }

    public static void main2(String[] args) throws InterruptedException, SigarException {
        /*String data = "[" +
                "{\"code\":\"cpuLoad\",\"name\":\"CPU负载\",\"frequency\":30}," +
                "{\"code\":\"cpuTemp\",\"name\":\"CPU温度\",\"frequency\":30}," +
                "{\"code\":\"mac\",\"name\":\"MAC地址\",\"frequency\":30}," +
                "{\"code\":\"memoryCap\",\"name\":\"内存容量\",\"frequency\":30}," +
                "{\"code\":\"diskCap\",\"name\":\"硬盘容量\",\"frequency\":30}," +
                "{\"code\":\"diskModel\",\"name\":\"硬盘型号\",\"frequency\":60}," +
                "]";
        List<ApqInfoDataVo> apqInfoDataVoss = new ArrayList<>();
        ApqInfoDataVo apqInfoDataVo = new ApqInfoDataVo();
        apqInfoDataVo.setCode("diskModel");
        apqInfoDataVo.setName("硬盘型号");
        apqInfoDataVo.setFrequency(60);
        apqInfoDataVoss.add(apqInfoDataVo);

        ApqInfoDataVo apqInfoDataVo2 = new ApqInfoDataVo();
        apqInfoDataVo2.setCode("memoryCap");
        apqInfoDataVo2.setName("内存容量");
        apqInfoDataVo2.setFrequency(30);
        apqInfoDataVoss.add(apqInfoDataVo2);

        String s = JSONUtil.toJsonStr(apqInfoDataVoss);
        System.out.println(s);

        List<ApqInfoDataVo> apqInfoDataVos = JSONUtil.toList(data, ApqInfoDataVo.class);
        List<Map> apqInfoDataVos2 = JSONUtil.toList(data, Map.class);*/

//        System.out.println(OshiUtil.getSensors().getCpuTemperature());
//        System.out.println(OshiUtil.getHardware().getDiskStores()[0].getModel());
//        System.out.println(DataSizeUtil.format(OshiUtil.getHardware().getDiskStores()[0].getReads()));

        //System.out.println("CPU内存使用率:" + new DecimalFormat("#.##%").format((OshiUtil.getMemory().getAvailable() * 1.0 / OshiUtil.getMemory().getTotal())));

        /*while (true) {
            //System.out.println("CPU负载:" + new DecimalFormat("#.##%").format(OshiUtil.getCpuInfo().getUsed() / 100));
            //System.out.println("硬盘剩余:" + OshiUtil.getHardware().getDiskStores()[0].getWriteBytes());

            System.out.println(getCpuUsed());

            Thread.sleep(3);
        }*/

    }

    public static double getCpuUsed() {
        /*SystemInfo systemInfo = new SystemInfo();
        CentralProcessor processor = systemInfo.getHardware().getProcessor();
        long[] prevTicks = processor.getSystemCpuLoadTicks();
        long[] ticks = processor.getSystemCpuLoadTicks();
        long nice = ticks[CentralProcessor.TickType.NICE.getIndex()] - prevTicks[CentralProcessor.TickType.NICE.getIndex()];
        long irq = ticks[CentralProcessor.TickType.IRQ.getIndex()] - prevTicks[CentralProcessor.TickType.IRQ.getIndex()];
        long softirq = ticks[CentralProcessor.TickType.SOFTIRQ.getIndex()] - prevTicks[CentralProcessor.TickType.SOFTIRQ.getIndex()];
        long steal = ticks[CentralProcessor.TickType.STEAL.getIndex()] - prevTicks[CentralProcessor.TickType.STEAL.getIndex()];
        long cSys = ticks[CentralProcessor.TickType.SYSTEM.getIndex()] - prevTicks[CentralProcessor.TickType.SYSTEM.getIndex()];
        long user = ticks[CentralProcessor.TickType.USER.getIndex()] - prevTicks[CentralProcessor.TickType.USER.getIndex()];
        long iowait = ticks[CentralProcessor.TickType.IOWAIT.getIndex()] - prevTicks[CentralProcessor.TickType.IOWAIT.getIndex()];
        long idle = ticks[CentralProcessor.TickType.IDLE.getIndex()] - prevTicks[CentralProcessor.TickType.IDLE.getIndex()];
        long totalCpu = user + nice + cSys + idle + iowait + irq + softirq + steal;
        //System.out.println("----------------cpu信息----------------");
        //System.out.println("cpu核数:" + processor.getLogicalProcessorCount());
        //System.out.println("cpu系统使用率:" + new DecimalFormat("#.##%").format(cSys * 1.0 / totalCpu));
        //System.out.println("cpu用户使用率:" + new DecimalFormat("#.##%").format(user * 1.0 / totalCpu));
        //System.out.println("cpu当前等待率:" + new DecimalFormat("#.##%").format(iowait * 1.0 / totalCpu));
        //System.out.println("cpu当前使用率:" + new DecimalFormat("#.##%").format(1.0-(idle * 1.0 / totalCpu)));

        return new BigDecimal((1.0 - (idle * 1.0 / totalCpu)) * 100).setScale(2, BigDecimal.ROUND_DOWN).doubleValue();*/

        CpuInfo cpuInfo = OshiUtil.getCpuInfo(100);
        double used = Double.parseDouble(new DecimalFormat("#.00").format(100 - cpuInfo.getFree()));
        return Math.min(Math.max(used, 0), 100);
    }

    private static void printDisks(List<HWDiskStore> list) {
        System.out.println("Disks:");
        for (HWDiskStore disk : list) {
            boolean readwrite = disk.getReads() > 0 || disk.getWrites() > 0;
            System.out.format(" %s: (model: %s - S/N: %s) size: %s, reads: %s (%s), writes: %s (%s), xfer: %s ms%n",
                    disk.getName(), disk.getModel(), disk.getSerial(),
                    disk.getSize() > 0 ? FormatUtil.formatBytesDecimal(disk.getSize()) : "?",
                    readwrite ? disk.getReads() : "?", readwrite ? FormatUtil.formatBytes(disk.getReadBytes()) : "?",
                    readwrite ? disk.getWrites() : "?", readwrite ? FormatUtil.formatBytes(disk.getWriteBytes()) : "?",
                    readwrite ? disk.getTransferTime() : "?");
            HWPartition[] partitions = disk.getPartitions();
            if (partitions == null) {
                // TODO Remove when all OS's implemented
                continue;
            }
            for (HWPartition part : partitions) {
                System.out.format(" |-- %s: %s (%s) Maj:Min=%d:%d, size: %s%s%n", part.getIdentification(),
                        part.getName(), part.getType(), part.getMajor(), part.getMinor(),
                        FormatUtil.formatBytesDecimal(part.getSize()),
                        part.getMountPoint().isEmpty() ? "" : " @ " + part.getMountPoint());
            }
        }
    }

    public static void main(String[] args) throws SigarException {
        Sigar sigar = new Sigar();
//        System.out.println(sigar.getCpu().toString());
//        org.hyperic.sigar.CpuInfo cpuInfo = sigar.getCpuInfoList()[0];
//        String cpuModelV = cpuInfo.getVendor() + " " + cpuInfo.getModel();
//        System.out.println(cpuModelV);

        Mem mem = sigar.getMem();
        String memoryAvailableV = FormatUtil.formatBytesDecimal(mem.getFree()) + "/" + FormatUtil.formatBytesDecimal(mem.getTotal())
                + "(" + Double.parseDouble(new DecimalFormat("#.00").format(mem.getFree() * 1.0 / mem.getTotal() * 100)) + ")";
        System.out.println(mem.getFree() * 1.0 / mem.getTotal());
        System.out.println(memoryAvailableV);
    }
}
