package com.priusis.client.service.core;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.Maps;
import com.google.common.io.Resources;
import com.priusis.client.data.kv.*;
import com.priusis.client.service.*;
import com.priusis.client.service.conf.*;
import com.priusis.client.service.data.*;
import com.priusis.controller.ProgramController;
import com.priusis.controller.RemoteControlController;
import com.priusis.lib.R;
import com.priusis.monitor.mqtt.*;
import com.priusis.service.common.MacAddrService;
import com.priusis.utils.MacAddrUtil;
import com.priusis.utils.SysConfigUtil;
import io.netty.buffer.ByteBuf;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.util.concurrent.Promise;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.hyperic.sigar.Mem;
import org.hyperic.sigar.Sigar;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import java.io.*;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.KeyStore;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.priusis.client.util.JsonTools.*;

/**
 * Created by priusis on 16.01.17.
 */
@Slf4j
public class MqttServiceImpl implements MqttService, MqttHandler, MqttClientCallback {

    @Value(value = "${apq.iot-gateway}")
    private String gateway;

    @Value(value = "${apq.url.device_info}")
    private String deviceInfo;

    @Value(value = "${apq.client.productId}")
    private Long productId;

    @Value(value = "${apq.client.version}")
    private String clientVersion;

    @Value(value = "${apq.client.tenantId}")
    private Long tenantId;

    @Value(value = "${apq.control.request.isTimeSet:true}")
    private boolean isTimeSet;


    //private static final String GATEWAY_RPC_TOPIC = "gateway/rpc";
    //private static final String GATEWAY_ATTRIBUTES_TOPIC = "gateway/attrs";
    //private static final String GATEWAY_TELEMETRY_TOPIC = "gateway/telemetry";
    //private static final String GATEWAY_REQUESTS_ATTRIBUTES_TOPIC = "gateway/attrs/req";
    //private static final String GATEWAY_RESPONSES_ATTRIBUTES_TOPIC = "gateway/attrs/res";
    //private static final String GATEWAY_CONNECT_TOPIC = "gateway/connect";
    //private static final String GATEWAY_DISCONNECT_TOPIC = "gateway/disconnect";
    //private static final String GATEWAY = "GATEWAY";

    private static final String DEVICE_TELEMETRY_TOPIC = "devices/telemetry";
    private static final String DEVICE_ATTRIBUTES_TOPIC = "devices/attrs";
    private static final String DEVICE_RPC_REQ_TOPIC = "devices/rpc/req";
    private static final String DEVICE_RPC_RES_TOPIC = "devices/rpc/res";
    private static final String DEVICE_RPC_TOPIC = DEVICE_RPC_REQ_TOPIC + "/+";
    private static final String DEVICE_EVENT_TOPIC = "devices/event";
    private static final String DEVICE_GET_ATTRIBUTES_REQUEST_TOPIC = "devices/attrs/req/1";
    private static final String DEVICE_GET_ATTRIBUTES_RESPONSE_PLUS_TOPIC = "devices/attrs/res/+";
    private static final String DEVICE_GET_ATTRIBUTES_RESPONSE_TOPIC = "devices/attrs/res/1";

    private static final String JKS = "JKS";
    private static final long DEFAULT_CONNECTION_TIMEOUT = 10000;
    private static final long DEFAULT_POLLING_INTERVAL = 1000;

    private final ConcurrentMap<String, DeviceInfo> devices = new ConcurrentHashMap<>();
    private final AtomicLong attributesCount = new AtomicLong();
    private final AtomicLong telemetryCount = new AtomicLong();
    private final AtomicInteger msgIdSeq = new AtomicInteger();
    private final Set<AttributesUpdateSubscription> attributeUpdateSubs = ConcurrentHashMap.newKeySet();
    private final Map<AttributeRequestKey, AttributeRequestListener> pendingAttrRequestsMap = new ConcurrentHashMap<>();

    private final String REQUEST_ID_PATTERN = "(?<requestId>\\d+)";
    private final String DEVICE_RPC_REQUEST_TOPIC_PATTERN = DEVICE_RPC_REQ_TOPIC + "/" + REQUEST_ID_PATTERN;
    private final Pattern DEVICE_RPC_REQUEST_PATTERN = Pattern.compile(DEVICE_RPC_REQUEST_TOPIC_PATTERN);

    private String tenantLabel;

    private PersistentFileService persistentFileService;

    private Consumer<String> extensionsConfigListener;
    private PcCoreConfiguration configuration;
    private PcConnectionConfiguration connection;
    private PcReportingConfiguration reporting;
    private PcPersistenceConfiguration persistence;


    private volatile ObjectNode error;
    private volatile Boolean realDataMonitor = false;
    Sigar sigar = new Sigar();
    private MqttClient tbClient;

    private ScheduledExecutorService scheduler;
    private ExecutorService mqttSenderExecutor;
    private ExecutorService mqttReceiverExecutor;
    private ExecutorService callbackExecutor = Executors.newCachedThreadPool();
    private ScheduledExecutorService realDataMonitorScheduler;
    private static ScheduledFuture<?> realDataMonitorSchedulerFuture;

    @Autowired
    private NioEventLoopGroup nioEventLoopGroup;

    @Autowired
    private MacAddrService macAddrService;

    @Resource
    private RemoteControlController remoteControlController;

    @Resource
    private ProgramController programController;

    private Locale enlocale = new Locale("en", "US");
    private DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(enlocale);

    public MqttServiceImpl(PcCoreConfiguration configuration, Consumer<String> extensionsConfigListener) {
        this.configuration = configuration;
        this.extensionsConfigListener = extensionsConfigListener;
    }


    @Override
    @PostConstruct
    public void init() {
        BlockingQueue<MessageFuturePair> incomingQueue = new LinkedBlockingQueue<>();
        this.tenantLabel = configuration.getLabel();
        this.connection = configuration.getConnection();
        this.reporting = configuration.getReporting();
        this.persistence = configuration.getPersistence();
        this.tenantLabel = configuration.getLabel();
        initTimeouts();

        scheduler = Executors.newSingleThreadScheduledExecutor();
        scheduler.submit(() -> {
            initMqttClient();
            initMqttSender(incomingQueue);
            initMqttReceiver(incomingQueue);

            realDataMonitorScheduler = Executors.newSingleThreadScheduledExecutor();

//            Map<String, Integer> mqttRpcDataMessageParam = Maps.newHashMap();
//            mqttRpcDataMessageParam.put("minutes", 2);
//            MqttRpcDataMessage mqttRpcDataMessage = MqttRpcDataMessage.builder().params(JSONUtil.toJsonStr(mqttRpcDataMessageParam)).build();
//            realDataMonitorScheduler(mqttRpcDataMessage);
        });

    }

    private void initTimeouts() {
        // Backwards compatibility with old config file
        if (connection.getConnectionTimeout() == 0) {
            connection.setConnectionTimeout(DEFAULT_CONNECTION_TIMEOUT);
        }
        if (persistence.getPollingInterval() == 0) {
            persistence.setPollingInterval(DEFAULT_POLLING_INTERVAL);
        }
    }

    @Override
    public void destroy() throws Exception {
        scheduler.shutdownNow();
        realDataMonitorScheduler.shutdownNow();
        callbackExecutor.shutdownNow();
        mqttSenderExecutor.shutdownNow();
        mqttReceiverExecutor.shutdownNow();
        tbClient.disconnect();
    }

    @Override
    public String getTenantLabel() {
        return tenantLabel;
    }


    @Override
    public MqttDeliveryFuture onDeviceAttributesUpdate(List<KvEntry> attributes) {
        final int msgId = msgIdSeq.incrementAndGet();
        log.trace("[{}] Updating device attributes: {}", msgId, attributes);
        ObjectNode node = newNode();
        //ObjectNode deviceNode = node.putObject(deviceName);
        attributes.forEach(kv -> putToNode(node, kv));
        final int packSize = attributes.size();
        return persistMessage(DEVICE_ATTRIBUTES_TOPIC, msgId, toNodeBytes(node),
                message -> {
                    log.debug("[{}] Device attributes were delivered!", msgId);
                    attributesCount.addAndGet(packSize);
                },
                error -> log.warn("[{}] Failed to report device attributes!", msgId, error));
    }

    @Override
    public MqttDeliveryFuture onDeviceEventUpdate(List<KvEntry> attributes) {
        final int msgId = msgIdSeq.incrementAndGet();
        log.trace("[{}] Updating device event: {}", msgId, attributes);
        ObjectNode node = newNode();
        //ObjectNode deviceNode = node.putObject(deviceName);
        attributes.forEach(kv -> putToNode(node, kv));
        final int packSize = attributes.size();
        return persistMessage(DEVICE_EVENT_TOPIC, msgId, toNodeBytes(node),
                message -> {
                    log.debug("[{}] Device event were delivered!", msgId);
                    attributesCount.addAndGet(packSize);
                },
                error -> log.warn("[{}] Failed to report device event!", msgId, error));
    }

    private MqttDeliveryFuture persistMessage(String topic,
                                              int msgId,
                                              byte[] payload,
                                              Consumer<Void> onSuccess,
                                              Consumer<Throwable> onFailure) {
        try {
            return persistentFileService.persistMessage(topic, msgId, payload, onSuccess, onFailure);
        } catch (Throwable e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void onDeviceAttributeRequest(AttributeRequest request, Consumer<AttributeResponse> listener) {
        final int msgId = msgIdSeq.incrementAndGet();
        String deviceName = request.getDeviceName();
        AttributeRequestKey requestKey = new AttributeRequestKey(request.getRequestId(), request.getDeviceName());
        log.trace("[{}][{}] Requesting {} attribute: {}", deviceName, msgId, request.isClientScope() ? "client" : "shared", request.getAttributeKey());

        ObjectNode node = newNode();
        node.put("id", request.getRequestId());
        node.put("client", request.isClientScope());
        node.put("device", request.getDeviceName());
        node.put("key", request.getAttributeKey());
        MqttMessage msg = new MqttMessage(toBytes(node));

        msg.setId(msgId);
        pendingAttrRequestsMap.put(requestKey, new AttributeRequestListener(request, listener));
        persistMessage(DEVICE_GET_ATTRIBUTES_REQUEST_TOPIC, msgId, toBytes(node),
                message -> {
                    log.debug("[{}] Device attributes request was delivered!", msgId);
                },
                error -> {
                    log.warn("[{}] Failed to report device attributes!", msgId, error);
                    pendingAttrRequestsMap.remove(requestKey);
                });
    }

    @Override
    public void onDeviceRpcResponse(RpcCommandResponse response) {
        int requestId = response.getRequestId();
        String bizRequestId = response.getBizRequestId();
        String data = response.getData();
        String method = response.getMethod();

        /*ObjectNode paramsNode = newNode();
        paramsNode.put("code", "1");
        paramsNode.put("msg", "success");*/
        byte[] msgData = toBytes(newNode().put("requestId", bizRequestId).put("method", method)
                .put("resTime", System.currentTimeMillis()).put("response", data));
        persistMessage(DEVICE_RPC_RES_TOPIC + "/" + requestId, msgIdSeq.incrementAndGet(), msgData, null,
                error ->
                        log.warn("Could not publish RPC res", error));

    }

    @Override
    public boolean subscribe(AttributesUpdateSubscription subscription) {
        return subscribe(attributeUpdateSubs::add, subscription);
    }

    @Override
    public boolean unsubscribe(AttributesUpdateSubscription subscription) {
        return unsubscribe(attributeUpdateSubs::remove, subscription);
    }

    @Override
    public void onError(Exception e) {
        onError(null, e);
    }

    @Override
    public void onError(String deviceName, Exception e) {
        ObjectNode node = newNode();
        node.put("ts", System.currentTimeMillis());
        if (deviceName != null) {
            node.put("device", deviceName);
        }
        node.put("error", toString(e));
        error = node;
    }

    @Override
    public void onMessage(String topic, ByteBuf payload) {
        String message = payload.toString(StandardCharsets.UTF_8);

        log.trace("Message arrived [{}] {}", topic, message);
        callbackExecutor.submit(() -> {
            try {
                if (topic.equals(DEVICE_ATTRIBUTES_TOPIC)) {
                    onAttributesUpdate(message);
                } else if (topic.equals(DEVICE_GET_ATTRIBUTES_RESPONSE_PLUS_TOPIC)) {
                    onDeviceAttributesResponse(message);
                } else if (topic.startsWith(DEVICE_RPC_REQ_TOPIC)) {
                    onRpcCommand(topic, message);
                } else if (topic.startsWith(DEVICE_ATTRIBUTES_TOPIC)) {
                    onGatewayAttributesUpdate(message);
                } else if (topic.equals(DEVICE_GET_ATTRIBUTES_RESPONSE_TOPIC)) {
                    onGatewayAttributesGet(message);
                }
            } catch (Exception e) {
                log.warn("Failed to process arrived message：{}!", message, e);
            }
        });

    }

    @Override
    public void connectionLost(Throwable throwable) {
        log.warn("Lost connection to Priusisiot.");
        pendingAttrRequestsMap.clear();
        devices.clear();
    }

    @Override
    public void onSuccessfulReconnect() {

    }

    private void onAttributesUpdate(String message) {
        JsonNode payload = fromString(message);
        String deviceName = payload.get("device").asText();
        Set<AttributesUpdateListener> listeners = attributeUpdateSubs.stream()
                .filter(sub -> sub.matches(deviceName)).map(sub -> sub.getListener())
                .collect(Collectors.toSet());
        if (!listeners.isEmpty()) {
            JsonNode data = payload.get("data");
            List<KvEntry> attributes = getKvEntries(data);
            listeners.forEach(listener -> callbackExecutor.submit(() -> {
                try {
                    listener.onAttributesUpdated(deviceName, attributes);
                } catch (Exception e) {
                    log.error("[{}] Failed to process attributes update", deviceName, e);
                }
            }));
        }
    }

    private void onRpcCommand(String topic, String message) {

        JsonNode payload = fromString(message);
        String bizRequestId = payload.get("requestId").asText();
        int requestId = 0;
        Matcher fwMatcher = DEVICE_RPC_REQUEST_PATTERN.matcher(topic);
        if (fwMatcher.find()) {
            requestId = Integer.parseInt(fwMatcher.group("requestId"));
        }
        MqttRpcDataMessage mqttRpcDataMessage = MqttRpcDataMessage.builder().requestId(requestId).bizRequestId(bizRequestId)
                .method(payload.get("method").asText()).sendTime(System.currentTimeMillis())
                .params(Optional.ofNullable(payload.get("params")).map(JsonNode::toString).orElse(null)).build();

        // 存储rpc下发的数据
        powerHandler(message, requestId, bizRequestId, mqttRpcDataMessage);
        remoteControlHandler(message, requestId, bizRequestId, mqttRpcDataMessage);

        try {
            if (!Arrays.asList("power", "remote_control", "remote_control_exit", "remote_control_accept",
                    "real_data_monitor", "facility_scan", "remote_rtc", "control_allow", "controll",
                    "control_port_allow", "BlackList_allow", "WhiteList_allow").contains(mqttRpcDataMessage.getMethod())) {
                persistentFileService.flushRpcDataToFile(mqttRpcDataMessage);
            }
        } catch (IOException e) {
            log.error("Failed to process flushRpcDataToFile : {}", message, e);
        }
    }

    private void powerHandler(String message, int requestId, String bizRequestId, MqttRpcDataMessage mqttRpcDataMessage) {
        String method = mqttRpcDataMessage.getMethod();
        if (!"power".equals(method)) {
            return;
        }
        try {
            if ("power".equals(method)) {
                Integer type = JSONUtil.parseObj(mqttRpcDataMessage.getParams()).getInt("powerType");
                if (type == 0) {
                    log.info("远程关机!!!");
                    Runtime.getRuntime().exec("cmd /c shutdown -s");
                } else if (type == 1) {
                    log.info("远程重启!!!");
                    Runtime.getRuntime().exec("cmd /c shutdown -r");
                }

                // response
                ObjectNode paramsNode = newNode();
                paramsNode.put("code", "1");
                paramsNode.put("msg", "success");
                paramsNode.put("status", "success");
                byte[] msgData = toBytes(newNode().put("requestId", bizRequestId).put("method", method)
                        .put("resTime", System.currentTimeMillis()).put("response", paramsNode.toString()));
                persistMessage(DEVICE_RPC_RES_TOPIC + "/" + requestId, msgIdSeq.incrementAndGet(), msgData, null,
                        error ->
                                log.warn("Could not publish power res", error));
            }
        } catch (Exception e) {
            log.error("Failed to process rpc command persistent : {}", message, e);
            ObjectNode paramsNode = newNode();
            paramsNode.put("code", "0");
            paramsNode.put("msg", "fail");
            paramsNode.put("status", "fail");
            byte[] msgData = toBytes(newNode().put("requestId", bizRequestId).put("method", method)
                    .put("resTime", System.currentTimeMillis()).put("response", paramsNode.toString()));
            persistMessage(DEVICE_RPC_RES_TOPIC + "/" + requestId, msgIdSeq.incrementAndGet(), msgData, null,
                    error ->
                            log.warn("Could not publish power res", error));
        }
    }





    /**
     * 远程RPC处理
     *
     * @param message
     * @param requestId
     * @param mqttRpcDataMessage
     */
    private void remoteControlHandler(String message, int requestId, String bizRequestId, MqttRpcDataMessage
            mqttRpcDataMessage) {
        String method = mqttRpcDataMessage.getMethod();
        if (!"remote_control".equals(method) && !"remote_control_exit".equals(method) && !"remote_control_accept".equals(method)) {
            return;
        }
        try {
            if ("remote_control".equals(method)) {// 远程控制请求
                log.info("收到远程控制请求!!!");
                if (StrUtil.equals("RUNNING", remoteControlController.vncServerIsRunning())) {
                    log.warn("VNC Server 已在运行中，忽略此次请求");
                    return;
                }
                if (isTimeSet) {
                    SysConfigUtil.saveProperty("control.request", "1");
                } else {
                    SysConfigUtil.saveProperty("control.request", "-1");
                    remoteControlController.startVncServer(0);
                }
            }
            if ("remote_control_exit".equals(method)) {
                log.info("退出远程控制!!!");
                remoteControlController.stopVncServer();
            }
            if ("remote_control_accept".equals(method)) {
                log.info("远程控制请求接受!!! params: {}", mqttRpcDataMessage.getParams());
                if (StrUtil.equals("RUNNING", remoteControlController.vncServerIsRunning())) {
                    log.warn("VNC Server 已在运行中，忽略此次请求");
                    return;
                } else {
                    Long minutes = Convert.toLong(mqttRpcDataMessage.getParams());
                    R<Void> startVncResult = remoteControlController.startVncServer(minutes);
                    log.info("远程控制请求接受!!! startVncResult: {}", startVncResult.getMsg());
                }
            }
            // response
            ObjectNode paramsNode = newNode();
            paramsNode.put("code", "1");
            paramsNode.put("msg", "success");
            paramsNode.put("status", "success");
            byte[] msgData = toBytes(newNode().put("requestId", bizRequestId).put("method", method)
                    .put("resTime", System.currentTimeMillis()).put("response", paramsNode.toString()));
            persistMessage(DEVICE_RPC_RES_TOPIC + "/" + requestId, msgIdSeq.incrementAndGet(), msgData, null,
                    error ->
                            log.warn("Could not publish remoteControl res", error));
        } catch (Exception e) {
            log.error("Failed to process rpc command persistent : {}", message, e);
            ObjectNode paramsNode = newNode();
            paramsNode.put("code", "0");
            paramsNode.put("msg", "fail");
            paramsNode.put("status", "fail");
            byte[] msgData = toBytes(newNode().put("requestId", bizRequestId).put("method", method)
                    .put("resTime", System.currentTimeMillis()).put("response", paramsNode.toString()));
            persistMessage(DEVICE_RPC_RES_TOPIC + "/" + requestId, msgIdSeq.incrementAndGet(), msgData, null,
                    error ->
                            log.warn("Could not publish remoteControl res", error));
        }
    }





    /**
     * 数据上报频率请求
     *
     * @param mqttRpcDataMessage
     */
    private void realDataMonitorScheduler(MqttRpcDataMessage mqttRpcDataMessage) {
        decimalFormat.applyPattern("#.00");
        // 持续时长
        Integer durationSeconds = JSONUtil.parseObj(mqttRpcDataMessage.getParams()).getInt("minutes", 1) * 60;
        Runnable runnable = new Runnable() {
            int i = 1;

            public void run() {
                try {
                    if (i >= durationSeconds) {
                        realDataMonitor = false;
                        realDataMonitorSchedulerFuture.cancel(false);
                    }
                    String cpuLoad = "cpuLoad";
                    String memoryOccupyRate = "memoryOccupyRate";
                    // CPU负载
                    double cpuLoadV = Double.parseDouble(decimalFormat.format(sigar.getCpuPerc().getCombined() * 100));
                    Mem mem = sigar.getMem();
                    // 内存占用率
                    double memoryOccupyRateV = Double.parseDouble(decimalFormat.format(mem.getUsed() * 1.0 / mem.getTotal() * 100));

                    Map<String, Object> data = MapUtil.<String, Object>builder()
                            .put(cpuLoad, cpuLoadV)
                            .put(memoryOccupyRate, memoryOccupyRateV)
                            .build();
                    log.info("采集数据，上报属性: i:{}, mapData:{}", i, data);
                    List<KvEntry> attrData = getKvEntries(fromString(JSONUtil.toJsonStr(data)));
                    DeviceData dd = new DeviceData(attrData);
                    if (dd != null) {
                        if (!dd.getAttributes().isEmpty()) {
                            onDeviceAttributesUpdate(dd.getAttributes());
                        }
                    }
                } catch (Throwable e) {
                    log.error("Real data monitor Runnable发生异常", e);
                } finally {
                    i++;
                }

            }
        };
        realDataMonitorSchedulerFuture = realDataMonitorScheduler.scheduleAtFixedRate(runnable, 0, 1, TimeUnit.SECONDS);
    }

    private void onGatewayAttributesGet(String message) {
        log.info("Configuration arrived! {}", message);
        JsonNode payload = fromString(message);
        if (payload.get("shared").get("configuration") != null) {
            String configuration = payload.get("shared").get("configuration").asText();
            if (!StringUtils.isEmpty(configuration)) {
                updateConfiguration(configuration);
            }
        }
    }

    private void onGatewayAttributesUpdate(String message) {
        log.info("Configuration updates arrived! {}", message);
        JsonNode payload = fromString(message);
        if (payload.has("configuration")) {
            String configuration = payload.get("configuration").asText();
            if (!StringUtils.isEmpty(configuration)) {
                updateConfiguration(configuration);
            }
        }
    }

    private void updateConfiguration(String configuration) {
        try {
            if (extensionsConfigListener != null) {
                extensionsConfigListener.accept(configuration);
            }
            onAppliedConfiguration(configuration);
        } catch (Exception e) {
            log.warn("Failed to update extension configurations [[]]", e.getMessage(), e);
        }
    }

    @Override
    public void onAppliedConfiguration(String configuration) {
        byte[] msgData = toBytes(newNode().put("appliedConfiguration", configuration));
        persistMessage(DEVICE_ATTRIBUTES_TOPIC, msgIdSeq.incrementAndGet(), msgData, null,
                error ->
                        log.warn("Could not publish applied configuration", error));
    }

    @Override
    public void onConfigurationError(Exception e, PcExtensionConfiguration configuration) {
        String id = configuration.getId();
        byte[] msgDataError = toBytes(newNode().put(id + "ExtensionError", toString(e)));
        persistMessage(DEVICE_TELEMETRY_TOPIC, msgIdSeq.incrementAndGet(), msgDataError, null,
                error -> log.warn("Could not report extension error", error));

        byte[] msgDataStatus = toBytes(newNode().put(id + "ExtensionStatus", "Failure"));
        persistMessage(DEVICE_TELEMETRY_TOPIC, msgIdSeq.incrementAndGet(), msgDataStatus, null,
                error -> log.warn("Could not report extension error status", error));
    }

    @Override
    public void onConfigurationStatus(String id, String status) {
        byte[] extentionStatusData = toBytes(newNode().put(id + "ExtensionStatus", status));
        persistMessage(DEVICE_TELEMETRY_TOPIC, msgIdSeq.incrementAndGet(), extentionStatusData,
                message -> log.info("Reported status [{}] of extension [{}]", status, id),
                error -> log.warn("Extension status reporting failed", error));


        byte[] extentionErrorData = toBytes(newNode().put(id + "ExtensionError", ""));
        persistMessage(DEVICE_TELEMETRY_TOPIC, msgIdSeq.incrementAndGet(), extentionErrorData,
                null, error ->
                        log.warn("Extension error clearing failed", error));
    }

    @Override
    public File flushRpcDataToFile(MqttRpcDataMessage mqttRpcDataMessage) throws IOException {
        return persistentFileService.flushRpcDataToFile(mqttRpcDataMessage);
    }

    @Override
    public MqttRpcDataMessage readFromFile(String method) throws IOException {
        return persistentFileService.readFromFile(method);
    }

    private void onDeviceAttributesResponse(String message) {
        JsonNode payload = fromString(message);
        AttributeRequestKey requestKey = new AttributeRequestKey(payload.get("id").asInt(), payload.get("device").asText());

        AttributeRequestListener listener = pendingAttrRequestsMap.get(requestKey);
        if (listener == null) {
            log.warn("[{}][{}] Can't find listener for request", requestKey.getDeviceName(), requestKey.getRequestId());
            return;
        }

        AttributeRequest request = listener.getRequest();
        AttributeResponse.AttributeResponseBuilder response = AttributeResponse.builder();

        response.requestId(request.getRequestId());
        response.deviceName(request.getDeviceName());
        response.key(request.getAttributeKey());
        response.clientScope(request.isClientScope());
        response.topicExpression(request.getTopicExpression());
        response.valueExpression(request.getValueExpression());

        String key = listener.getRequest().getAttributeKey();
        JsonNode value = payload.get("value");
        if (value == null) {
            response.data(Optional.empty());
        } else if (value.isBoolean()) {
            response.data(Optional.of(new BooleanDataEntry(key, value.asBoolean())));
        } else if (value.canConvertToLong()) {
            response.data(Optional.of(new LongDataEntry(key, value.asLong())));
        } else if (value.isDouble()) {
            response.data(Optional.of(new DoubleDataEntry(key, value.asDouble())));
        } else {
            response.data(Optional.of(new StringDataEntry(key, value.asText())));
        }

        callbackExecutor.submit(() -> {
            try {
                listener.getListener().accept(response.build());
            } catch (Exception e) {
                log.error("[{}][{}] Failed to process attributes response", requestKey.getDeviceName(), requestKey.getRequestId(), e);
            }
        });
    }

    private void initMqttSender(BlockingQueue<MessageFuturePair> incomingQueue) {
        mqttSenderExecutor = Executors.newSingleThreadExecutor();
        mqttSenderExecutor.submit(new MqttMessageSender(persistence, connection, tbClient, this, persistentFileService, incomingQueue));
    }

    private void initMqttReceiver(BlockingQueue<MessageFuturePair> incomingQueue) {
        mqttReceiverExecutor = Executors.newSingleThreadExecutor();
        mqttReceiverExecutor.submit(new MqttMessageReceiver(persistentFileService, incomingQueue, connection.getIncomingQueueWarningThreshold()));
    }

    private static String toString(Exception e) {
        StringWriter sw = new StringWriter();
        e.printStackTrace(new PrintWriter(sw));
        return sw.toString();
    }

    private <T> boolean subscribe(Function<T, Boolean> f, T sub) {
        if (f.apply(sub)) {
            log.info("Subscription added: {}", sub);
            return true;
        } else {
            log.warn("Subscription was already added: {}", sub);
            return false;
        }
    }

    private <T> boolean unsubscribe(Function<T, Boolean> f, T sub) {
        if (f.apply(sub)) {
            log.info("Subscription removed: {}", sub);
            return true;
        } else {
            log.warn("Subscription was already removed: {}", sub);
            return false;
        }
    }

    private MqttClient initMqttClient() {
        try {
            MqttClientConfig mqttClientConfig = getMqttClientConfig();
            if (connection.getSecurity().isApiByMac()) {
                MqttClientSecurityConfiguration clientSecurityByMac = getClientSecurityByMac();
                if (null != clientSecurityByMac) {
                    mqttClientConfig.setUsername(clientSecurityByMac.getAccessToken());
                } else {
                    log.info("no user please reg");
                    mqttClientConfig.setUsername("");
                }
            } else {
                mqttClientConfig.setUsername(connection.getSecurity().getAccessToken());
            }

            tbClient = MqttClient.create(mqttClientConfig, this);
            tbClient.setCallback(this);
            tbClient.setEventLoop(nioEventLoopGroup);
            log.debug("try connect client");
            connectTbClient();

            if (tbClient.isConnected()) {
                //tbClient.on(DEVICE_ATTRIBUTES_TOPIC, this).await(connection.getConnectionTimeout(), TimeUnit.MILLISECONDS);
                tbClient.on(DEVICE_GET_ATTRIBUTES_RESPONSE_PLUS_TOPIC, this).await(connection.getConnectionTimeout(), TimeUnit.MILLISECONDS);
                tbClient.on(DEVICE_GET_ATTRIBUTES_RESPONSE_TOPIC, this).await(connection.getConnectionTimeout(), TimeUnit.MILLISECONDS);
                tbClient.on(DEVICE_RPC_TOPIC, this).await(connection.getConnectionTimeout(), TimeUnit.MILLISECONDS);

                byte[] msgData = toBytes(newNode().put("sharedKeys", "configuration"));
                persistMessage(DEVICE_GET_ATTRIBUTES_REQUEST_TOPIC, msgIdSeq.incrementAndGet(), msgData,
                        null,
                        error -> log.warn("Error getiing attributes", error));
            }
            return tbClient;
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);
            Thread.currentThread().interrupt();
            return null;
        } catch (ExecutionException e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e);
        }
    }

    private void connectTbClient() throws InterruptedException, ExecutionException {
        // 获取client username
        String username = tbClient.getClientConfig().getUsername();
        if (StrUtil.isBlank(username)) {
            MqttClientSecurityConfiguration clientSecurityByMac = getClientSecurityByMac();
            if (null == clientSecurityByMac) {
                log.info("no user please reg");
                log.info("Priusisiot Gateway connection failed. Reconnecting in [{}] milliseconds", connection.getRetryInterval());
                Thread.sleep(connection.getRetryInterval());
                connectTbClient();
            } else {
                tbClient.getClientConfig().setUsername(clientSecurityByMac.getAccessToken());
            }
        }
        Promise<MqttConnectResult> connectResult = (Promise<MqttConnectResult>) tbClient.connect(connection.getHost(), connection.getPort());
        connectResult.addListener(future -> {
            if (future.isSuccess()) {
                MqttConnectResult result = (MqttConnectResult) future.getNow();
                log.debug("Gateway connect result code: [{}]", result.getReturnCode());
            } else {
                log.error("Unable to connect to mqtt server!");
                if (future.cause() != null) {
                    log.error(future.cause().getMessage(), future.cause());
                }
            }
        });
        try {
            MqttConnectResult mqttConnectResult = connectResult.get(connection.getConnectionTimeout(), TimeUnit.MILLISECONDS);
            if (!mqttConnectResult.isSuccess()) {
                log.error("Unable to sucess connect to mqtt server try again!");
                tbClient.disconnect();
            }
        } catch (TimeoutException e) {
            log.error("Unable to connect to mqtt server try again!");
            String message = "Unable to connect to Priusisiot. Connection timed out after [" + connection.getConnectionTimeout() + "] milliseconds";
            log.error(message, e);
            Thread.sleep(connection.getRetryInterval());
            connectTbClient();
        }
    }

    private MqttClientConfig getMqttClientConfig() {
        MqttClientConfig mqttClientConfig;
        if (connection.getSecurity().isApiByMac()) {
            mqttClientConfig = new MqttClientConfig();
            MqttClientSecurityConfiguration clientSecurityByMac = getClientSecurityByMac();
            if (null != clientSecurityByMac) {
                mqttClientConfig.setUsername(clientSecurityByMac.getAccessToken());
            } else {
                log.info("no user please reg");
                mqttClientConfig.setUsername("");
            }
        } else {
            if (!StringUtils.isEmpty(connection.getSecurity().getAccessToken())) {
                if (StringUtils.isEmpty(connection.getSecurity().getTruststore())) {
                    mqttClientConfig = new MqttClientConfig();
                    mqttClientConfig.setUsername(connection.getSecurity().getAccessToken());
                } else {
                    try {
                        SslContext sslCtx = initOneWaySslContext(connection.getSecurity());
                        mqttClientConfig = new MqttClientConfig(sslCtx);
                    } catch (Exception e) {
                        log.error(e.getMessage(), e);
                        throw new RuntimeException(e);
                    }
                }
            } else {
                try {
                    SslContext sslCtx = initTwoWaySslContext(connection.getSecurity());
                    mqttClientConfig = new MqttClientConfig(sslCtx);
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                    throw new RuntimeException(e);
                }
            }
        }
        return mqttClientConfig;
    }

    private SslContext initOneWaySslContext(MqttClientSecurityConfiguration configuration) throws Exception {
        URL tsUrl = Resources.getResource(configuration.getTruststore());
        File tsFile = new File(tsUrl.toURI());

        TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        KeyStore trustStore = KeyStore.getInstance(JKS);
        try (InputStream tsFileInputStream = new FileInputStream(tsFile)) {
            trustStore.load(tsFileInputStream, configuration.getTruststorePassword().toCharArray());
        }
        tmf.init(trustStore);

        return SslContextBuilder.forClient().trustManager(tmf).build();
    }

    private SslContext initTwoWaySslContext(MqttClientSecurityConfiguration configuration) throws Exception {
        URL ksUrl = Resources.getResource(configuration.getKeystore());
        File ksFile = new File(ksUrl.toURI());
        URL tsUrl = Resources.getResource(configuration.getTruststore());
        File tsFile = new File(tsUrl.toURI());

        TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        KeyStore trustStore = KeyStore.getInstance(JKS);
        try (InputStream tsFileInputStream = new FileInputStream(tsFile)) {
            trustStore.load(tsFileInputStream, configuration.getTruststorePassword().toCharArray());
        }
        tmf.init(trustStore);

        KeyStore keyStore = KeyStore.getInstance(JKS);
        try (InputStream ksFileInputStream = new FileInputStream(ksFile)) {
            keyStore.load(ksFileInputStream, configuration.getKeystorePassword().toCharArray());
        }
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        kmf.init(keyStore, configuration.getKeystorePassword().toCharArray());

        return SslContextBuilder.forClient().keyManager(kmf).trustManager(tmf).build();
    }

    public void setPersistentFileService(PersistentFileService persistentFileService) {
        this.persistentFileService = persistentFileService;
    }

    private MqttClientSecurityConfiguration getClientSecurityByMac() {
        String mac = null;
        MqttRpcDataMessage mqttRpcDataMessage = null;
        try {
            mqttRpcDataMessage = readFromFile("mac_addr");
            mac = mqttRpcDataMessage.getParams();
        } catch (IOException e) {
            log.warn("未获取到MAC地址 FROM mac_addr");
        }
        try {
            if (null == mqttRpcDataMessage) {
                mac = MacAddrUtil.getSigarMac();
            }
            if (StrUtil.isBlank(mac)) {
                log.warn("未获取到MAC地址");
                return null;
            }

            String allMacAddress = MacAddrUtil.getAllMacAddress();

            log.info("获取到MAC地址:{}，{}", mac, allMacAddress);


            Map<String, Object> params = Maps.newHashMap();
            params.put("productId", productId);
            params.put("tenantId", tenantId);
            params.put("macAddress", mac);
            params.put("allMacAddress", allMacAddress);
            params.put("clientVersion", clientVersion);

            /*Map<String, String> map = System.getenv();
            String computerName = map.get("COMPUTERNAME");// 获取计算机名*/
            params.put("name", "测试机器");

//            String userName = map.get("USERNAME");// 获取用户名
//            params.put("name", computerName + "-" + userName);

            String remoteDeviceVoJson = HttpUtil.post(gateway + deviceInfo, JSONUtil.toJsonStr(params), 5000);
            log.info("remoteDeviceVoJson: {}", remoteDeviceVoJson);
            RemoteDeviceVoResult remoteDeviceVo = JSONUtil.toBean(remoteDeviceVoJson, RemoteDeviceVoResult.class);
            MqttClientSecurityConfiguration mqttClientSecurityConfiguration = new MqttClientSecurityConfiguration();
            mqttClientSecurityConfiguration.setAccessToken(remoteDeviceVo.getData().getSecretKey());

            String serverMac = remoteDeviceVo.getData().getMacAddress();
            MqttRpcDataMessage mqttRpcDataMessaget = MqttRpcDataMessage.builder()
                    .sendTime(System.currentTimeMillis())
                    .method("mac_addr")
                    .params(serverMac).build();
            flushRpcDataToFile(mqttRpcDataMessaget);
            macAddrService.initMacAddr(serverMac);

            return mqttClientSecurityConfiguration;

        } catch (Exception e) {
            log.error("获取Mac地址注册设备信息异常, mac:{}", mac, e);
        }
        //MqttClientSecurityConfiguration mqttClientSecurityConfiguration = new MqttClientSecurityConfiguration();
        //mqttClientSecurityConfiguration.setAccessToken("D6qjl0xNnDMtgSPMczGt");
        //return mqttClientSecurityConfiguration;
        return null;
    }

    public static void main(String[] args) {
        ObjectNode paramsNode = newNode();
        paramsNode.put("code", "0");
        paramsNode.put("msg", "fail");
        paramsNode.put("status", "fail");

        JSONObject jsonObject = JSONUtil.parseObj(paramsNode.toPrettyString());
        System.out.println(jsonObject.getStr("msg"));
    }
}
