package com.priusis.controller;

import cn.hutool.core.convert.Convert;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.priusis.client.service.TenantManagerService;
import com.priusis.lib.R;
import com.priusis.utils.SysConfigUtil;
import com.priusis.client.service.MqttRpcDataMessage;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author yangli
 * @since 2021/09/10
 */
@Slf4j
@Validated
@RestController
@RequestMapping(value = "config")
public class ConfigController {

    @Autowired
    private TenantManagerService service;

    @Value(value = "${apq.webrtc_path}")
    private String webrtcPath;

    @GetMapping(value = "get_download_path")
    public R<String> getDownloadPath() throws IOException {
        String path = SysConfigUtil.getProperty("download.path",
                System.getProperty("user.dir").substring(0, 3) + "apqdownloads");
        File dir = new File(path);
        if (!dir.exists()) dir.mkdirs();
        return R.success(path);
    }

    @GetMapping(value = "get_webrtc_path")
    public R<String> getWebRtcPath() {
        return R.success(webrtcPath);
    }

    @PostMapping(value = "save_download_path")
    public R<Void> saveDownloadPath(@RequestBody Map<String, String> params) throws IOException {
        String path = params.get("path");
        SysConfigUtil.saveProperty("download.path", path);
        return R.success();
    }

    @PostMapping(value = "save_auto_upgrade")
    public R<Void> saveAutoUpgrade(@RequestBody Map<String, String> params) throws IOException {
        SysConfigUtil.saveProperty("auto.upgrade", params.get("enable"));
        return R.success();
    }

    @GetMapping(value = "get_auto_upgrade")
    public R<Boolean> getAutoUpgrade() throws IOException {
        return R.success(Convert.toBool(SysConfigUtil.getProperty("auto.upgrade"), true));
    }

    @GetMapping(value = "get_device_sn")
    public R<String> getDeviceSN() throws IOException {
        return R.success(SysConfigUtil.getProperty("device.sn"));
    }

    @PostMapping(value = "save_device_sn")
    public R<Void> saveDeviceSN(@RequestBody Map<String, String> params) throws IOException {
        String sn = params.get("sn");
        SysConfigUtil.saveProperty("device.sn", sn);
        return R.success();
    }

    @GetMapping(value = "get_data_reporting")
    public R<JSONArray> getDataReporting() throws IOException {
        JSONArray defaultConf = JSONUtil.parseArray("[" +
                "{\"code\":\"operatingSystem\",\"name\":\"操作系统\",\"frequency\":60}," +
                "{\"code\":\"baseboard\",\"name\":\"主板\",\"frequency\":60}," +
                "{\"code\":\"ip\",\"name\":\"ip\",\"frequency\":60}," +
                "{\"code\":\"cpuLoad\",\"name\":\"CPU负载\",\"frequency\":60}," +
                "{\"code\":\"cpuTemp\",\"name\":\"CPU温度\",\"frequency\":60}," +
                "{\"code\":\"cpuModel\",\"name\":\"CPU型号\",\"frequency\":60}," +
                "{\"code\":\"gpuTemp\",\"name\":\"GPU温度\",\"frequency\":60}," +
                "{\"code\":\"mac\",\"name\":\"MAC地址\",\"frequency\":60}," +
                "{\"code\":\"ethernet\",\"name\":\"网络网卡\",\"frequency\":60}," +
                "{\"code\":\"memoryAvailable\",\"name\":\"内存剩余\",\"frequency\":60}," +
                "{\"code\":\"memoryCap\",\"name\":\"内存容量\",\"frequency\":60}," +
                "{\"code\":\"memoryOccupyRate\",\"name\":\"内存占用率\",\"frequency\":60}," +
                "{\"code\":\"diskModel\",\"name\":\"磁盘型号\",\"frequency\":60}," +
                "{\"code\":\"diskCap\",\"name\":\"磁盘容量\",\"frequency\":60}," +
                "{\"code\":\"diskSpeed\",\"name\":\"磁盘传输速度\",\"frequency\":60}," +
                "{\"code\":\"diskTemp\",\"name\":\"磁盘温度\",\"frequency\":60}," +
                "{\"code\":\"diskFree\",\"name\":\"磁盘剩余量\",\"frequency\":60}" +
                "]");
        JSONArray conf = JSONUtil.parseArray(SysConfigUtil.getProperty("data.reporting", "[]"));
        Set<String> set = conf.stream().map(JSONObject.class::cast).map(o -> o.getStr("code")).collect(Collectors.toSet());
        defaultConf.stream().map(JSONObject.class::cast).filter(o -> !set.contains(o.getStr("code"))).forEach(conf::add);
        return R.success(conf);
    }

    @PostMapping(value = "save_data_reporting")
    public R<Void> saveDataReporting(@RequestBody JSONArray array) throws IOException {
        String dataReport = array.toString();
        SysConfigUtil.saveProperty("data.reporting", dataReport);
        // 存储配置作为Rpc下发的数据
        try {
            MqttRpcDataMessage mqttRpcDataMessage = MqttRpcDataMessage.builder()
                    .method("info").params(dataReport).build();
            service.flushRpcDataToFile(mqttRpcDataMessage);
        } catch (IOException e) {
            log.error("Failed to save_data_reporting persistent : {}", dataReport);
        }
        return R.success();
    }


    @PostConstruct
    public void init() {
        try {
            // 初始化自动保存一次数据上报设置
            saveDataReporting(getDataReporting().getData());
        } catch (Exception e) {
            log.error("初始化异常", e);
        }
    }
}
